//-----------------------------------------------------------------------------------------------------------
// File Control Program
//   (c)2010  Kenji Arai / JH1PJL
//      http://www.page.sannet.ne.jp/kenjia/index.html   
//          March 27th,2010  Started
//          March 28th,2010  
//-----------------------------------------------------------------------------------------------------------
// Function
//      5 channeles ADC data records into a file which is located in mbed storage
//      If USE_LCD = 1, data shows on a Text LCD
//      If USE_RTC = 1, time stamp also writes in the file
// Connection
//      Analog input    PIN 15,16,17,19,20
//      LCD             PIN 22,23,24,25,26,27,28
//       -> CAUTION!! pin assignment is different
//          with " http://mbed.org/projects/cookbook/wiki/TextLCD "
//      RTC             PIN 3 needs to connect 3V Battery
//       -> Please refer my program " RTC_w_COM" for time adjustment
//          at " http://mbed.org/users/kenjiArai/programs/RTC_w_COM/5yi9a/ "
// IMPORTANT ISSUE!!
//      Once you start this program
//        then MBED storage on PC (recogunition as "Memory strage") will disappear
//      After finish the program (stop at end of main(), fclose()),
//        you can see MBED again
//      If you want to stop the program, please push "Reset Button" on mbed board
//      --> You can keep push "Reset Button" and rewite new bin file
//-----------------------------------------------------------------------------------------------------------
#include "mbed.h"
#include "TextLCD.h"

#define DEBUG 1             // 1= Shows progress on PC via USB ( virtual COM line)
#define USE_LCD 1           // 1= Display the data on LCD
#define USE_RTC 1           // 1= Use RTC (need 3V supply and time adjustment before use)

#define NO_OF_SAMPLE 100    // Total recording length -> unit=sec
#define TIM_INTVL 10        // Insert time stamp in the file every ?? sec

DigitalOut myled(LED1);     // Indicate the sampling period

#if USE_LCD
TextLCD lcd(p22, p28, p27, p26, p25, p24, p23, 40, 2); // rs,rw,e,d0,d1,d2,d3,40char's x 2 lines
#endif

AnalogIn ain_G_X(p15);      // G Sensor
AnalogIn ain_G_Y(p16);      // G Sensor
AnalogIn ain_G_Z(p17);      // G Sensor
AnalogIn ain_BAT(p19);      // Battery Volt
AnalogIn ain_TEMP(p20);     // Temperature Sensor

LocalFileSystem local("local"); // Create the local filesystem under the name "local"

int main() {
    char buf[40];           // data buffer for text
    int i;                  // count for number of record
    float x,y,z,b,t;        // Analog data
    #if USE_RTC
    time_t seconds, old_sec;// RTC data based on seconds
    int j;
    #endif
    
    // Open the file
    #if USE_RTC
    seconds = time(NULL);
    sprintf(buf,"/local/%d_out.txt",seconds); // File name is defined based on time from 1970/1/1   
    FILE *fp = fopen(buf, "w");               // Open "out.txt" on the local file system for writing
    fprintf(fp, "This is a test program for logging\r\n");
    #else
    FILE *fp = fopen("/local/out.txt", "w");  // Open "out.txt" on the local file system for writing
    fprintf(fp, "This is a test program for logging\r\n");    
    #endif
    
    #if DEBUG
    printf("\r\nStart sampling\r\n");
    #endif
    #if USE_LCD
    lcd.cls();
    #endif
    
    i = 0;
    #if USE_RTC
    seconds = time(NULL);
    old_sec = seconds; 
    strftime(buf,40, "%I:%M:%S %p (%Y/%m/%d)\r\n", localtime(&seconds));
    fprintf(fp,buf);
    #endif
    while(1) {
        // check time interval (1sec)
        #if USE_RTC
        myled = 1;
        wait(0.5);
        myled = 0;
        while ((seconds = time(NULL)) == old_sec) ;    // Wait 1 sec for loop
        old_sec = seconds;
        #else
        myled = 1;
        wait(0.5);
        myled = 0;
        wait(0.5);   
        #endif
        // Get analog data from each port
        x=ain_G_X.read();
        y=ain_G_Y.read();
        z=ain_G_Z.read();
        b=ain_BAT.read();
        t=ain_TEMP.read();
        // Write data into the file
        sprintf(buf, "G-Sen, %f, %f, %f  \r\n", x, y, z);
        fprintf(fp,buf);
        #if USE_LCD
        lcd.locate(0, 0);   // 1st line top
        lcd.printf(buf);
        #endif    
        sprintf(buf, "VB, %f, T, %f  \r\n", b, t);
        fprintf(fp,buf);
        #if USE_LCD
        lcd.locate(0, 1);   // 2nd line top
        lcd.printf(buf);
        #endif
        // if reach to expected data number then finsh
        if (++i >= NO_OF_SAMPLE){
            break;
        }
        // Set time satmp
        #if USE_RTC
        j = i / TIM_INTVL;
        if (i == j * TIM_INTVL){
            seconds = time(NULL);  
            strftime(buf,40, "Time, %I:%M:%S %p, (%Y/%m/%d)  \r\n", localtime(&seconds));
            fprintf(fp, buf);
        }
        #endif
        #if DEBUG
        printf("Sampling #%d / end=%d\r\n", i, NO_OF_SAMPLE);
        #endif           
    }
    fclose(fp);
    // for debug
    #if DEBUG
    printf("\r\nFinished sampling\r\n");
    #endif    
}
