//----------------------------------------------------------
//  Simultanuous AD Conversion by polling using
//  ADC2 and ADC3 on STM32F446 ---- Header
//
//  STM32F446 の ADC2, ADC3 を使って同時に AD 変換を開始し，
//  ポーリングによりアナログ信号を入力するクラス（ヘッダ）
//      A0 (PA_0) :  ADC2 CH0, 左
//      A1 (PA_1) :  ADC3 CH1, 右
//  Read(), Write() の引数：
//      第一引数：A0 (左)，第二引数：A1 (右)
//
//  2017/01/30, Copyright (c) 2017 MIKAMI, Naoki
//----------------------------------------------------------
/*
 *  Modified by Kenji Arai
 *      http://www.page.sannet.ne.jp/kenjia/index.html
 *      http://mbed.org/users/kenjiArai/
 *
 *      Started:  Feburary  1st, 2017
 *      Revised:  Feburary  1st, 2017
 */

#include "mbed.h"

#ifndef STM32F446xx
#error Select NUCLEO-F446RE.
#endif

#include "F446_DAC.hpp"

namespace Mikami
{
#ifndef F446_ADC_DUAL_HPP
#define F446_ADC_DUAL_HPP

    class AdcDual
    {
    public:
        // Constructor
        //      frequency: 標本化周波数
        explicit AdcDual(int frequency);

        // -1.0f <= ad1, ad2 <= 1.0f
        //      ad1: left, ad2: right
        virtual void Read(float &ad1, float &ad2);

        // 0 <= ad1, ad2 <= 4095
        //      ad1: left, ad2: right
        virtual void Read(uint16_t &ad1, uint16_t &ad2);

    protected:
        float ToFloat(uint16_t x)
        {   return AMP_*(x - 2048); }
    
    private:
        static const float AMP_ = 1.0f/2048.0f;
        static const uint32_t EOC23_ = ADC_CSR_EOC2 | ADC_CSR_EOC3;
        
        // AD 変換が完了するまで待つ
        void WaitDone()
        {   while((ADC->CSR & EOC23_) != EOC23_); }

        // AD 変換器の外部トリガに使うタイマ (TIM8) の設定
        void SetTim8(int frequency);

        // for inhibition of copy constructor
        AdcDual(const AdcDual&);
        // for inhibition of substitute operator
        AdcDual& operator=(const AdcDual&);     
    };

#endif  // F446_ADC_DUAL_HPP

#ifndef F446_ADC_SINGLE_HPP
#define F446_ADC_SINGLE_HPP

    class AdcSingle
    {
    public:
        // Constructor
        //      frequency: sampling frequency
        explicit AdcSingle(int frequency);

        virtual void Read(float &ad);

        virtual void Read(uint16_t &ad);

    protected:
        float ToFloat(uint16_t x)
        {   return AMP_*(x - 2048); }
    
    private:
        static const float AMP_ = 1.0f/2048.0f;
        static const uint32_t EOC2_ = ADC_CSR_EOC2;
        
        // waiting ADC conversion
        void WaitDone()
        {   while((ADC->CSR & EOC2_) != EOC2_); }

        // Set timer(TIM() for trigger for ADC
        void SetTim8(int frequency);

        // for inhibition of copy constructor
        AdcSingle(const AdcSingle&);
        // for inhibition of substitute operator
        AdcSingle& operator=(const AdcSingle&);     
    };

#endif  // F446_ADC_SINGLE_HPP
}

