/*
 * mbed library program 
 *  Texas Instruments / DRV8830 H-Bridge Voltage-Controlled Motor Driver
 *      http://www.ti.com/product/drv8830
 *
 * Copyright (c) 2014,'17 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created: August      6th, 2014
 *      Revised: August     23rd, 2017
 */

#ifndef        MBED_DRV8830
#define        MBED_DRV8830

// DRV8830 LOW-VOLTAGE MOTOR DRIVER WITH SERIAL INTERFACE
//  7bit address = 0b01100000(0x60) -> 8bit = 0b11000000(0xc0) -> 0xc1(Read) or 0xc0(Write)
//  ADDR_01 = (A1=0)+(A0=1), ADDR_1N = (A1=1)+(A0=No Connection)
//      -> Please make sure your H/W configuration
#define DRV8830ADDR_00          0xc0
#define DRV8830ADDR_0N          0xc2
#define DRV8830ADDR_01          0xc4
#define DRV8830ADDR_N0          0xc6
#define DRV8830ADDR_NN          0xc8
#define DRV8830ADDR_N1          0xca
#define DRV8830ADDR_10          0xcc
#define DRV8830ADDR_1N          0xce
#define DRV8830ADDR_11          0xd0

//  fault status
#define DRV8830_F_ILIMIT        0x10
#define DRV8830_F_OTS           0x08
#define DRV8830_F_UVLO          0x04
#define DRV8830_F_OCP           0x02
#define DRV8830_F_FAULT         0x01

/** DRV8830 class
 *
 *  This is a driver code for the DRV8830 H-Bridge Voltage-Controlled Motor Driver.
 *    http://www.ti.com/product/drv8830
 *
 * @code
 * #include "mbed.h"
 * #include "DRV8830.h"
 *
 * // I2C Communication
 * DRV8830 motor(PinName p_sda, PinName p_scl, addr);
 * // If you connected I2C line not only this device but also other devices,
 * //     you need to declare following method.
 * I2C     i2c(PinName p_sda, PinName p_scl);
 * DRV8830 motor(I2C& p_i2c, addr);
 * 
 * int main() {
 * uint8_t status;
 *    
 *     //  Speed control
 *     motor.speed(0.5);    // CW
 *     motor.speed(-0.5);   // CCW
 *     motor.speed(0.0);    // Stop(FREE)
 *     //  Check error and reset
 *     status = motor.status();
 *     if (status & DRV8830_F_FAULT){
 *        motor.reset();
 *     } 
 * }
 *  @endcode
 */

class DRV8830 {
public:
    /** Configure data pin
      * @param data SDA and SCL pins
      * @param DRV8830 address (H/W configuration of A1,A0)
      */
    DRV8830(PinName p_sda, PinName p_scl, uint8_t addr);
    
    /** Configure data pin (with other devices on I2C line)
      * @param I2C previous definition
      * @param DRV8830 address (H/W configuration of A1,A0)
      */
    DRV8830(I2C& p_i2c, uint8_t addr);

    /** Set motor speed
      * @param speed (Range 0 to 1.0 or 0 to -1.0)
      * @return none
      */
    void speed(float speed);

    /** Set voltage
      * @param voltage (0 to 5V)
      * @return none
      */
    void set_voltage(float volt);
   
    /** Read status
      * @param none
      * @return FAULT register content
      */
    uint8_t status();

    /** Reset DRV8830 chip
      * @param none
      * @return none
      */
    void reset();

protected:
    I2C *_i2c_p;
    I2C &_i2c;

private:
    char DRV8830_addr;
};

#endif  //  MBED_DRV8830

