/*
 * Mbed Application program / check RTC (Real Time Clock)
 *
 * Copyright (c) 2017,'18,'19 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  https://os.mbed.com/users/kenjiArai/
 *      Created:    August    11th, 2017
 *      Revised:    October   14th, 2019
 */

#include "select_program.h"
//#define EXAMPLE_1_CHECK_RTC
#ifdef EXAMPLE_1_CHECK_RTC

//  Include --------------------------------------------------------------------
#include "mbed.h"

//  Definition -----------------------------------------------------------------
#define     USER_SW_ON      1

//  Constructor ----------------------------------------------------------------
DigitalIn   userSW(BUTTON1);
DigitalOut  myled(LED1);                // Indicate the sampling period
Serial pc(USBTX, USBRX, 115200);

//  RAM ------------------------------------------------------------------------

//  ROM / Constant data --------------------------------------------------------
const char * msg0  = "Is a time correct? If no, please hit any key. ";
const char * msg1  = "<Push USER SW then enter sleep mode> ";
const char * msg2  = "\r\nEnter Standby Mode, please push RESET to wake-up\r\n";

//  Function prototypes --------------------------------------------------------
static void goto_standby(void);
static void time_enter_mode(void);
static void chk_and_set_time(char *ptr);
static int  xatoi (char **str, unsigned long *res);
static void get_line (char *buff, int len);

//------------------------------------------------------------------------------
//  Control Program
//------------------------------------------------------------------------------
int main()
{
    char buf[64];               // data buffer for text
    time_t seconds;
    uint8_t wait_counter = 0;

    pc.printf("\x1b[2J\x1b[H %s\r\n %s %s (UTC)\r\n",
              __FILE__, __DATE__, __TIME__);
    printf(" RTC EXAMPLE FOR DISCO-F769NI:\r\n");
    myled = !myled;
    ThisThread::sleep_for(1000);
    myled = !myled;
    ThisThread::sleep_for(1000);
    while(1) {
        seconds = time(NULL);
        strftime(buf, 50, " %B %d,'%y, %H:%M:%S,", localtime(&seconds));
        pc.printf("[Time] %s", buf);
        pc.printf(" sec=%d\r\n", seconds);
        pc.printf("%s", msg0);
        pc.printf("%s\r", msg1);
        wait_counter = 0;
        while (seconds == time(NULL)) {
            if (pc.readable()) {
                buf[0] = pc.getc();  // dummy read
                time_enter_mode();
            }
            if (userSW == USER_SW_ON) {
                pc.printf("%s", msg2);
                ThisThread::sleep_for(1000);
                myled = 0;
                goto_standby();
            }
            ThisThread::sleep_for(50);
            if (++wait_counter > (2000 / 50)) {
                break;
            }
        }
        uint8_t n = strlen(msg0) + strlen(msg1);
        for (uint8_t i = 0; i < n; i++) {
            pc.putc(' ');
        }
        pc.printf("      \r"); // Not use '\n'
        myled = !myled;
    }
}

void goto_standby(void)
{
    ThisThread::sleep_for(0x7fffffff);    // sleep 24 days
}

void time_enter_mode(void)
{
    char *ptr;
    char linebuf[64];

    pc.printf("\r\nSet time into RTC\r\n");
    pc.printf(" e.g. >19 10 14 10 11 12 -> October 14th,'19, 10:11:12\r\n");
    pc.printf(" If time is fine, just hit enter\r\n");
    pc.putc('>');
    ptr = linebuf;
    get_line(ptr, sizeof(linebuf));
    pc.printf("\r");
    chk_and_set_time(ptr);
}

//  Get key input data
void get_line (char *buff, int len)
{
    char c;
    int idx = 0;

    for (;;) {
        c = pc.getc();
        if (c == '\r') {
            buff[idx++] = c;
            break;
        }
        if ((c == '\b') && idx) {
            idx--;
            pc.putc(c);
            pc.putc(' ');
            pc.putc(c);
        }
        if (((uint8_t)c >= ' ') && (idx < len - 1)) {
            buff[idx++] = c;
            pc.putc(c);
        }
    }
    buff[idx] = 0;
    pc.putc('\n');
}

void chk_and_set_time(char *ptr)
{
    unsigned long p1;
    struct tm t;
    time_t seconds;

    if (xatoi(&ptr, &p1)) {
        t.tm_year       = (uint8_t)p1 + 100;
        pc.printf("Year:%d ",(int)p1);
        xatoi( &ptr, &p1 );
        t.tm_mon        = (uint8_t)p1 - 1;
        pc.printf("Month:%d ",(int)p1);
        xatoi( &ptr, &p1 );
        t.tm_mday       = (uint8_t)p1;
        pc.printf("Day:%d ",(int)p1);
        xatoi( &ptr, &p1 );
        t.tm_hour       = (uint8_t)p1;
        pc.printf("Hour:%d ",(int)p1);
        xatoi( &ptr, &p1 );
        t.tm_min        = (uint8_t)p1;
        pc.printf("Min:%d ",(int)p1);
        xatoi( &ptr, &p1 );
        t.tm_sec        = (uint8_t)p1;
        pc.printf("Sec:%d \r\n",(int)p1);
    } else {
        return;
    }
    seconds = mktime(&t);
    set_time(seconds);
    pc.printf(
        "Date: %04d/%02d/%02d, %02d:%02d:%02d\r\n",
        t.tm_year + 1900, t.tm_mon + 1, t.tm_mday,
        t.tm_hour, t.tm_min, t.tm_sec
    );
}

//  Change string -> integer
int xatoi (char **str, unsigned long *res)
{
    unsigned long val;
    unsigned char c, radix, s = 0;

    while ((c = **str) == ' ') {
        (*str)++;
    }
    if (c == '-') {
        s = 1;
        c = *(++(*str));
    }
    if (c == '0') {
        c = *(++(*str));
        if (c <= ' ') {
            *res = 0;
            return 1;
        }
        if (c == 'x') {
            radix = 16;
            c = *(++(*str));
        } else {
            if (c == 'b') {
                radix = 2;
                c = *(++(*str));
            } else {
                if ((c >= '0')&&(c <= '9')) {
                    radix = 8;
                } else {
                    return 0;
                }
            }
        }
    } else {
        if ((c < '1')||(c > '9')) {
            return 0;
        }
        radix = 10;
    }
    val = 0;
    while (c > ' ') {
        if (c >= 'a') {
            c -= 0x20;
        }
        c -= '0';
        if (c >= 17) {
            c -= 7;
            if (c <= 9) {
                return 0;
            }
        }
        if (c >= radix) {
            return 0;
        }
        val = val * radix + c;
        c = *(++(*str));
    }
    if (s) {
        val = -val;
    }
    *res = val;
    return 1;
}

#endif
