/*
 * Mbed Application program
 *
 * Copyright (c) 2018 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Modify:     March     21st, 2018
 *      Revised:    April     30th, 2018
 */

//  Include --------------------------------------------------------------------
#include    "mbed.h"
#include    "USBHostMSD.h"
#include    "FATFileSystem.h"
#include    "mon.h"
#include    <stdlib.h>

//  Definition -----------------------------------------------------------------

//  Object ---------------------------------------------------------------------
DigitalOut  led(LED1);
DigitalIn   user_sw(USER_BUTTON);
Serial      pc(USBTX,USBRX);
USBHostMSD  msd;
DigitalOut  usb_5v_line(PB_2, 1);

//  RAM ------------------------------------------------------------------------
uint8_t     usb_memory_status;
time_t      seconds;
// ADC
float       v5;
float       v3r3;

bool running = true;

//  ROM / Constant data --------------------------------------------------------
char *const opngmsg =
    ""__FILE__ "\r\n"__DATE__ " " __TIME__ " (UTC)\r\n""\r\n";

//  Function prototypes --------------------------------------------------------
void save_logging_data(uint8_t flg);
void msd_task(void const *);
void mon(void);

//------------------------------------------------------------------------------
//  Control Program
//------------------------------------------------------------------------------
int main()
{
    pc.puts(opngmsg);
    Thread msdTask(msd_task, NULL, osPriorityNormal, 1024 * 4);
    while(true) {
        led=!led;
        Thread::wait(500);
        if (running == false) {
            mon();
            NVIC_SystemReset();
        }
    }
}

void msd_task(void const *)
{
    int i = 0;
    FATFileSystem fs("usb");
    int err;

    pc.printf("wait for usb memory stick insertion\r\n");
    while(1) {

        // try to connect a MSD device
        while(!msd.connect()) {
            Thread::wait(500);
        }
        if (fs.mount(&msd) != 0) {
            continue;
        } else {
            pc.printf("file system mounted\r\n");
        }
        if  (!msd.connect()) {
            continue;
        }
        running = true;
        // if device disconnected, try to connect again
        while (msd.connected()) {
            save_logging_data(1);
            if (user_sw == 1) {
                break;
            }
            if (pc.readable()) {
                char c = pc.getc(); // dummy read
                mon();
            }
            Thread::wait(500);
        }
        running = false;
        while (msd.connected()) {
            Thread::wait(500);
        }
        while (fs.unmount() < 0) {
            Thread::wait(500);
            pc.printf("unmount\r\n");
        }
    }
}

// Save several data into the USB Memory
void save_logging_data(uint8_t flg)
{
    char buf[64];   // data buffer for text
    FILE *fp;

    if (!msd.connect()) {
        error("USB Flash drive not found.\r\n");
        usb_memory_status = 1;
    }
    fp = fopen("/usb/mydata.txt", "a");
    if (flg) {
        pc.printf("/usb/mydata.txt\r\n");
    }
    if(fp == NULL) {
        usb_memory_status = 1;
        if (flg) {
            pc.printf( "\r\n Could not open file for write\r\n");
        }
    }
    for (int i = 0; 16 > i; i++) {
        uint32_t size = get_disk_freespace();
        pc.printf("free %u kB, ", size);
        //v3r3 = VREF_VOLT / a_vref.read();
        v3r3 = 3.321f;
        //v5 = a_5v.read() * v3r3 * FCT_V5;
        v5 = 5.012f;
        sprintf(buf, "$DATA0,");
        fprintf(fp,buf);
        // RTC
        seconds = time(NULL);
        strftime(buf, 64, "RTC,%H:%M:%S,%Y/%m/%d,", localtime(&seconds));
        if (flg) {
            pc.printf("%s", buf);
        }
        fprintf(fp,buf);
        // Write speration ',' into the file
        //            123456789012345678901234567890
        sprintf(buf, ",,,,,,,,,,,,,,,,,,,,,,,,,,,,");
        fprintf(fp,buf);
        // Write data into the file
        sprintf(buf, "5V,%6.3f,V3R3,%6.3f,", v5, v3r3);
        fprintf(fp,buf);
        if (flg) {
            pc.printf("Volt, %s", buf);
        }
        sprintf(buf, "GPS,%d,USB,%d,CAM,%d,CNT,%04d\r\n",
                0, usb_memory_status, 1, i);
        fprintf(fp,buf);
        fprintf(fp,buf);
        if (flg) {
            pc.printf("%s", buf);
        }
    }
    fclose(fp);
    if (flg) {
        pc.printf("\r\n");
    }
}

