/*
 * Mbed-OS5 & OS-2 --> Added OS6
 *  Check EventTimer(on OS6, change name Ticker) behavior
 *
 * Copyright (c) 2020,'21 Kenji Arai / JH1PJL
 *  http://www7b.biglobe.ne.jp/~kenjia/
 *  https://os.mbed.com/users/kenjiArai/
 *      Created:    May       12th, 2020
 *      Revised:    March     30th, 2021
 */
/*
    test result target = 50uS
        Nucleo-L432KC
        2.162
        0= 223, 1=  51, 2=  49, 3=  50, 4=  51, 5=  49, 6=  50, 7=  51, 8=  49
        5.15.3
        0= 109, 1=  50, 2=  50, 3=  48, 4=  51, 5=  52, 6=  47, 7=  52, 8=  48
        6.8.0
        0= 613, 1=  51, 2=  50, 3=  51, 4=  50, 5=  51, 6=  50, 7=  51, 8=  50

        Nucleo-F446RE
        2.162
        0=  85, 1=  50, 2=  50, 3=  50, 4=  50, 5=  50, 6=  50, 7=  50, 8=  50
        5.15.3
        0= 136, 1=  50, 2=  50, 3=  50, 4=  50, 5=  50, 6=  50, 7=  50, 8=  50
        6.8.0
        0= 138, 1=  50, 2=  50, 3=  50, 4=  50, 5=  50, 6=  50, 7=  50, 8=  50

        FRDM-K64F
        2.162
        0=  61, 1=  51, 2=  49, 3=  51, 4=  49, 5=  51, 6=  49, 7=  51, 8=  49
        5.15.3
        0=  61, 1=  53, 2=  49, 3=  48, 4=  49, 5=  53, 6=  49, 7=  49, 8=  50
        6.8.0
        0=  66, 1=  51, 2=  48, 3=  50, 4=  51, 5=  51, 6=  51, 7=  47, 8=  50


 */

#include "mbed.h"

#if (MBED_MAJOR_VERSION == 6)
class FlexTicker: public Ticker
#else
class FlexTicker: public TimerEvent
#endif
{
public:
    void attach(void(*fptr)(void)) { _function = Callback<void()>(fptr);}
    void detach() { remove();}
    void setNext(int delay) { insert(event.timestamp + delay);}
    void prime() { event.timestamp = us_ticker_read();}
protected:
    virtual void handler() { _function.call();}
    unsigned int _delay;
    Callback<void()> _function;
};

DigitalOut led(LED1);
FlexTicker tk;

#define TARGET_TIME     50      // 50us

volatile uint32_t num;
uint32_t time_data[10];

// call by TimerEvent
void pulse_out(void)
{
    led = 1;
    num++;
    time_data[num] = us_ticker_read();
    if (num < 10) {
        tk.setNext(TARGET_TIME);
    }
    led = 0;
}

int main()
{
    printf("MBED_MAJOR_VERSION = %d, ", MBED_MAJOR_VERSION);
    printf("MINOR = %d, ", MBED_MINOR_VERSION);
    printf("PATCH = %d\r\n", MBED_PATCH_VERSION);
    tk.attach(&pulse_out);
    while(true) {
        led = 1;
        num = 0;
        time_data[0] = us_ticker_read();
        tk.prime();
        led = 0;
        tk.setNext(TARGET_TIME);
#if (MBED_MAJOR_VERSION == 6)
        ThisThread::sleep_for(500ms);
#elif (MBED_MAJOR_VERSION == 5)
        thread_sleep_for(500);
#else
        wait_ms(500);
#endif
        for (uint32_t i = 0; i < 9; i++) {
            printf("%d=%4d, ", i, time_data[i+1] - time_data[i]);
        }
        printf("(<- No=time[us] (target %dus))\r\n", TARGET_TIME);
    }
}
