/*
 * Mbed Application program
 *  Check Standby Mode
 *
 * Copyright (c) 2020,'21 Kenji Arai / JH1PJL
 *  http://www7b.biglobe.ne.jp/~kenjia/
 *  https://os.mbed.com/users/kenjiArai/
 *      Revised:    March     12th, 2020
 *      Revised:    January   17th, 2021
 */

/*
    Reference information:
        https://forums.mbed.com/t/how-to-deep-sleep/7551
        https://os.mbed.com/users/kenjiArai/code/Check_DeepSleep_os5/
        https://os.mbed.com/users/kenjiArai/code/Check_StandBy_os5/

    DEEP SLEEP MODE (Tested on with mbed-os6.6.0)
        https://os.mbed.com/users/kenjiArai/code/Check_DeepSleep_os6/
        Nucleo-L152RE       -> 4.22uA (Normal run = 4mA to 7mA)
        Nucleo-L476RG       -> 2.23uA (Normal run = 7mA to 10mA)
        Nucleo-F411RE       -> 1.65mA (not uA)(Normal run = 7mA to 10mA)
        Nucleo-F446RE       -> 1.76mA (not uA)(Normal run = 14mA to 16mA)
    STANDBY MODE (Tested on with mbed-os6.6.0)
        Nucleo-L152RE       -> 3.90uA (Normal run = 4mA to 7mA)
        Nucleo-L476RG       -> 0.62uA (Normal run = 7mA to 10mA)
        Nucleo-F411RE       -> 3.40uA (Normal run = 7mA to 9mA)
        Nucleo-F446RE       -> 3.42uA (Normal run = 14mA to 16mA)
        DISCO-L475VG-IOT01A -> 0.58uA (Normal run = 7mA to 9mA)

        Current Measurement:
         Nucleo board has IDD Jumper (JP6).
         I measured CPU current using Digital Multi-meter DCI mode.
 */

//  Include --------------------------------------------------------------------
#include "mbed.h"
#include "WakeUp.h"

//  Definition -----------------------------------------------------------------

//  Constructor ----------------------------------------------------------------
DigitalIn   my_sw(USER_BUTTON);
#if defined(TARGET_DISCO_L475VG_IOT01A)
DigitalOut  myled(LED2,0);
#else
DigitalOut  myled(LED1,1);
#endif
static BufferedSerial pc(USBTX, USBRX, 9600);
AnalogIn    a_in(A0);
Timer       t;

//  RAM ------------------------------------------------------------------------

//  ROM / Constant data --------------------------------------------------------

//  Function prototypes --------------------------------------------------------
static void time_enter_mode(void);
static void chk_and_set_time(char *ptr);
static int32_t xatoi (char **str, int32_t *res);
static void get_line (char *buff, int32_t len);
extern void print_revision(void);

//------------------------------------------------------------------------------
//  Control Program
//------------------------------------------------------------------------------
int main()
{
    time_t seconds;
    char buf[64];
    uint32_t t_pass = 0;
    uint32_t loop_count = 1;
    float ain;

    printf("\r\nCheck current consumption at Standby mode.\r\n");
    print_revision();
    seconds = time(NULL);
    while (my_sw == 0) {;}
    ThisThread::sleep_for(10ms);
    while (true) {
        t.reset();
        t.start();
        if ((my_sw == 0) || (loop_count > 10)) {
            DigitalIn dmy0(LED1);
            DigitalIn dmy1(USBTX);
            DigitalIn dmy2(USBRX);
            WakeUp::standby_then_reset(20000);
            while(true) {;} // never executing this line
        }
        ain = a_in.read();
        myled = !myled;
        seconds = time(NULL);
        strftime(buf, 50, "%H:%M:%S -> ", localtime(&seconds));
        printf("%s", buf);
        printf(
            "analog = %4.3f, processing time=%3d, counter=%4d\r\n",
            ain, t_pass, loop_count++
        );
        t_pass = chrono::duration_cast<chrono::milliseconds>(
                               t.elapsed_time()).count();
        ThisThread::sleep_for(chrono::milliseconds(1000 - t_pass));
        if (pc.readable()) {
            strftime(buf, 50, " %B %d,'%y, %H:%M:%S\r\n", localtime(&seconds));
            printf("[Time] %s\r\n", buf);
            time_enter_mode();
        }
    }
}


static void time_enter_mode(void)
{
    char *ptr;
    char linebuf[64];

    if (pc.readable()) {
        pc.read(linebuf, 1);    // dummy read
    }
    puts("\r\nSet time into RTC");
    puts(" e.g. >21 1 12 13 14 15 -> January 12,'21, 13:14:14");
    linebuf[0] = '>';
    pc.write(linebuf, 1);
    ptr = linebuf;
    get_line(ptr, sizeof(linebuf));
    puts("\r");
    chk_and_set_time(ptr);
}

//  Change string -> integer
static int32_t xatoi(char **str, int32_t *res)
{
    uint32_t val;
    uint8_t c, radix, s = 0;

    while ((c = **str) == ' ') (*str)++;
    if (c == '-') {
        s = 1;
        c = *(++(*str));
    }
    if (c == '0') {
        c = *(++(*str));
        if (c <= ' ') {
            *res = 0;
            return 1;
        }
        if (c == 'x') {
            radix = 16;
            c = *(++(*str));
        } else {
            if (c == 'b') {
                radix = 2;
                c = *(++(*str));
            } else {
                if ((c >= '0')&&(c <= '9')) {
                    radix = 8;
                }   else {
                    return 0;
                }
            }
        }
    } else {
        if ((c < '1')||(c > '9')) {
            return 0;
        }
        radix = 10;
    }
    val = 0;
    while (c > ' ') {
        if (c >= 'a') c -= 0x20;
        c -= '0';
        if (c >= 17) {
            c -= 7;
            if (c <= 9) return 0;
        }
        if (c >= radix) return 0;
        val = val * radix + c;
        c = *(++(*str));
    }
    if (s) val = -val;
    *res = val;
    return 1;
}

//  Get key input data
static void get_line(char *buff, int32_t len)
{
    char c;
    char bf[8];
    int32_t idx = 0;

    for (;;) {
        pc.read(bf, 1);
        c = bf[0];
        //printf("0x%x \r\n", c);
        if (c == '\r') {
            buff[idx++] = c;
            break;
        }
        if ((c == '\b') && idx) {
            idx--;
            const char bf_bs[] =
            {0x1b, '[', '1', 'D', ' ', 0x1b, '[', '1', 'D'};
            pc.write(bf_bs, 9);
        }
        if (((uint8_t)c >= ' ') && (idx < len - 1)) {
            buff[idx++] = c;
            pc.write(bf, 1);
        }
    }
    buff[idx] = 0;
    bf[0] = '\n';
    pc.write(bf, 1);
}

//  Check key input strings and set time
static void chk_and_set_time(char *ptr)
{
    int32_t p1;
    struct tm t;
    time_t seconds;

    if (xatoi(&ptr, &p1)) {
        t.tm_year       = (uint8_t)p1 + 100;
        printf("Year:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_mon        = (uint8_t)p1 - 1;
        printf("Month:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_mday       = (uint8_t)p1;
        printf("Day:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_hour       = (uint8_t)p1;
        printf("Hour:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_min        = (uint8_t)p1;
        printf("Min:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_sec        = (uint8_t)p1;
        printf("Sec: %d \r\n",p1);
    } else {
        return;
    }
    seconds = mktime(&t);
    set_time(seconds);
    // Show Time with several example
    // ex.1
    printf(
        "Date: %04d/%02d/%02d, %02d:%02d:%02d\r\n",
        t.tm_year + 1900, t.tm_mon + 1, t.tm_mday, t.tm_hour, t.tm_min, t.tm_sec
    );
    char buf[64];
    // ex.2
    strftime(buf, 40, "%x %X", localtime(&seconds));
    printf("Date: %s\r\n", buf);
    // ex.3
    strftime(buf, 40, "%I:%M:%S %p (%Y/%m/%d)", localtime(&seconds));
    printf("Date: %s\r\n", buf);
    // ex.4
    strftime(buf, 40, "%B %d,'%y, %H:%M:%S", localtime(&seconds));
    printf("Date: %s\r\n", buf);
}
