/*
 * Mbed Application program
 *  Check StandBy Mode
 *
 * Copyright (c) 2020,'21 Kenji Arai / JH1PJL
 *  http://www7b.biglobe.ne.jp/~kenjia/
 *  https://os.mbed.com/users/kenjiArai/
 *      Revised:    March     12th, 2020
 *      Revised:    January   17th, 2020
 */

/*
    Reference information:
        https://forums.mbed.com/t/how-to-deep-sleep/7551
        https://os.mbed.com/users/kenjiArai/code/Check_DeepSleep_os6/
        https://os.mbed.com/users/kenjiArai/code/Check_StandBy_os6/

    DEEP SLEEP MODE (Tested on with mbed-os5.15.1)
        https://os.mbed.com/users/kenjiArai/code/Check_DeepSleep_os5/
        Nucleo-L152RE       -> 4.23uA (Normal run = 5mA to 8mA)
        Nucleo-L476RG       -> 2.13uA (Normal run = 7mA to 10mA)
        Nucleo-F411RE       -> 1.91mA (not uA)(Normal run = 7mA to 10mA)
        Nucleo-F446RE       -> 1.67mA (not uA)(Normal run = 14mA to 17mA)
    STANDBY MODE (Tested on with mbed-os5.15.1)
        Nucleo-L152RE       -> 3.89uA (Normal run = 5mA to 8mA)
        Nucleo-L476RG       -> 0.62uA (Normal run = 7mA to 10mA)
        Nucleo-F411RE       -> 3.35uA (Normal run = 7mA to 10mA)
        Nucleo-F446RE       -> 3.42uA (Normal run = 14mA to 17mA)
        DISCO-L475VG-IOT01A -> 0.58uA (Normal run = 7mA to 9mA)

        Current Measurement:
         Nucleo board has IDD Jumper (JP6).
         I measured CPU current using Digital Multi-meter DCI mode.
 */

//  Include --------------------------------------------------------------------
#include "mbed.h"
#include "WakeUp.h"

//  Definition -----------------------------------------------------------------

//  Constructor ----------------------------------------------------------------
DigitalIn   my_sw(USER_BUTTON);
#if defined(TARGET_DISCO_L475VG_IOT01A)
DigitalOut  myled(LED2,0);
#else
DigitalOut  myled(LED1,1);
#endif
Serial      pc(USBTX, USBRX);
AnalogIn    a_in(A0);
Timer       t;

//  RAM ------------------------------------------------------------------------

//  ROM / Constant data --------------------------------------------------------

//  Function prototypes --------------------------------------------------------
void time_enter_mode(void);
void chk_and_set_time(char *ptr);
int32_t  xatoi (char **str, int32_t *res);
void get_line (char *buff, int32_t len);
void print_revision(void);

//------------------------------------------------------------------------------
//  Control Program
//------------------------------------------------------------------------------
int main()
{
    time_t seconds;
    char buf[64];
    uint32_t t_pass = 0;
    uint32_t loop_count = 1;
    float ain;

    printf("\r\nCheck current consumption at Deep-sleep mode.\r\n");
    print_revision();
    seconds = time(NULL);
    while (my_sw == 0) {;}
    ThisThread::sleep_for(10);
    while (true) {
        t.reset();
        t.start();
        if ((my_sw == 0) || (loop_count > 10)) {
            DigitalIn dmy0(LED1);
            DigitalIn dmy1(USBTX);
            DigitalIn dmy2(USBRX);
            WakeUp::standby_then_reset(10000);
            while(true) {;} // never executing this line
        }
        ain = a_in.read();
        myled = !myled;
        seconds = time(NULL);
        strftime(buf, 50, "%H:%M:%S -> ", localtime(&seconds));
        pc.printf("%s", buf);
        pc.printf(
            "analog = %4.3f, processing time=%3d, counter=%4d\r\n",
            ain, t_pass, loop_count++
        );
        t_pass = t.read_ms();
        ThisThread::sleep_for(1000 - t_pass);
        if (pc.readable()) {
            strftime(buf, 50, " %B %d,'%y, %H:%M:%S\r\n", localtime(&seconds));
            pc.printf("[Time] %s\r\n", buf);
            time_enter_mode();
        }
    }
}

void time_enter_mode(void)
{
    char *ptr;
    char linebuf[64];

    if (pc.readable()) {
        pc.getc();    // dummy read
    }
    pc.printf("\r\nSet time into RTC\r\n");
    pc.printf(" e.g. >21 1 12 13 14 15 -> January 12,'21, 13:14:14\r\n");
    pc.putc('>');
    ptr = linebuf;
    get_line(ptr, sizeof(linebuf));
    pc.printf("\r");
    chk_and_set_time(ptr);
}

void get_line (char *buff, int len)
{
    char c;
    uint32_t idx = 0;

    while(true) {
        c = pc.getc();
        if (c == '\r') {
            buff[idx++] = c;
            break;
        }
        if ((c == '\b') && idx) {
            idx--;
            pc.putc(c);
            pc.putc(' ');
            pc.putc(c);
        }
        if (((uint8_t)c >= ' ') && (idx < len - 1)) {
            buff[idx++] = c;
            pc.putc(c);
        }
    }
    buff[idx] = 0;
    pc.puts("\r\n");
}

void chk_and_set_time(char *ptr)
{
    int32_t p1;
    struct tm t;
    time_t seconds;

    if (xatoi(&ptr, &p1)) {
        t.tm_year       = (uint8_t)p1 + 100;
        pc.printf("Year:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_mon        = (uint8_t)p1 - 1;
        pc.printf("Month:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_mday       = (uint8_t)p1;
        pc.printf("Day:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_hour       = (uint8_t)p1;
        pc.printf("Hour:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_min        = (uint8_t)p1;
        pc.printf("Min:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_sec        = (uint8_t)p1;
        pc.printf("Sec: %d \r\n",p1);
    } else {
        return;
    }
    seconds = mktime(&t);
    set_time(seconds);
    pc.printf(
        "Date: %04d/%02d/%02d, %02d:%02d:%02d\r\n",
        t.tm_year + 1900, t.tm_mon + 1, t.tm_mday, t.tm_hour, t.tm_min, t.tm_sec
    );
}

int32_t xatoi (char **str, int32_t *res)
{
    int32_t val;
    uint8_t c, radix, s = 0;

    while ((c = **str) == ' ') {
        (*str)++;
    }
    if (c == '-') {
        s = 1;
        c = *(++(*str));
    }
    if (c == '0') {
        c = *(++(*str));
        if (c <= ' ') {
            *res = 0;
            return 1;
        }
        if (c == 'x') {
            radix = 16;
            c = *(++(*str));
        } else {
            if (c == 'b') {
                radix = 2;
                c = *(++(*str));
            } else {
                if ((c >= '0')&&(c <= '9')) {
                    radix = 8;
                } else {
                    return 0;
                }
            }
        }
    } else {
        if ((c < '1')||(c > '9')) {
            return 0;
        }
        radix = 10;
    }
    val = 0;
    while (c > ' ') {
        if (c >= 'a') {
            c -= 0x20;
        }
        c -= '0';
        if (c >= 17) {
            c -= 7;
            if (c <= 9) {
                return 0;
            }
        }
        if (c >= radix) {
            return 0;
        }
        val = val * radix + c;
        c = *(++(*str));
    }
    if (s) {
        val = -val;
    }
    *res = val;
    return 1;
}

