/*
 * Mbed Application program
 *  Check Deep Sleep Mode
 *
 * Copyright (c) 2020,'21 Kenji Arai / JH1PJL
 *  http://www7b.biglobe.ne.jp/~kenjia/
 *  https://os.mbed.com/users/kenjiArai/
 *      Revised:    March     12th, 2020
 *      Revised:    January   15th, 2021
 */

/*
    Reference information:
        https://os.mbed.com/docs/mbed-os/v6.6/apis/power-management-sleep.html
        https://forums.mbed.com/t/how-to-deep-sleep/7551

    Tested on with mbed-os5.15.1
        https://os.mbed.com/users/kenjiArai/code/Check_DeepSleep_os5/
        Nucleo-L152RE       -> 4.23uA (Normal run = 5mA to 8mA)
        Nucleo-L476RG       -> 2.13uA (Normal run = 7mA to 10mA)
        Nucleo-F411RE       -> 1.91mA (not uA)(Normal run = 7mA to 10mA)
        Nucleo-F446RE       -> 1.67mA (not uA)(Normal run = 14mA to 17mA)
    Tested on with mbed-os6.6.0
        Nucleo-L152RE       -> 4.22uA (Normal run = 4mA to 7mA)
        Nucleo-L476RG       -> 2.23uA (Normal run = 7mA to 10mA)
        Nucleo-F411RE       -> 1.65mA (not uA)(Normal run = 7mA to 10mA)
        Nucleo-F446RE       -> 1.76mA (not uA)(Normal run = 14mA to 16mA)

        Current Measurement:
         Nucleo board has IDD Jumper (JP6).
         I measured CPU current using Digital Multi-meter DCI mode.
 */

//  Include --------------------------------------------------------------------
#include "mbed.h"

//  Definition -----------------------------------------------------------------
#define MODE    1   // MODE=3 does NOT work due to wait_ms() is abolished 

//  Constructor ----------------------------------------------------------------
DigitalIn   my_sw(USER_BUTTON);
DigitalOut  myled(LED1,1);
static BufferedSerial pc(USBTX, USBRX, 9600);
AnalogIn    a_in(A0);
Timer       t;

//  RAM ------------------------------------------------------------------------

//  ROM / Constant data --------------------------------------------------------

//  Function prototypes --------------------------------------------------------
void sw_irq(void);
void LowPowerConfiguration(void);
void time_enter_mode(void);
void chk_and_set_time(char *ptr);
int32_t  xatoi (char **str, int32_t *res);
void get_line (char *buff, int32_t len);
void print_revision(void);

//------------------------------------------------------------------------------
//  Control Program
//------------------------------------------------------------------------------
int main()
{
    time_t seconds;
    char buf[64];
    uint32_t t_pass = 0;
    uint32_t loop_count = 1;
    float ain;

    printf("\r\nCheck current consumption at Deep-sleep mode.\r\n");
    print_revision();
#if   MODE == 1
    printf("-->MODE=1 --> Use ThisThread::sleep_for() function.\r\n");
#elif MODE == 2
    printf("-->MODE=2 --> Use thread_sleep_for() function.\r\n");
#elif MODE == 3
    printf("-->MODE=3 --> Use wait_ms() function.\r\n");
#endif
    seconds = time(NULL);
    while (my_sw == 0) {;}
    ThisThread::sleep_for(10ms);
    while (true) {
        t.reset();
        t.start();
        if ((my_sw == 0) || (loop_count > 10)) {
            LowPowerConfiguration();
            InterruptIn my_irq(USER_BUTTON);
            while (my_irq.read() == 0) {;}
            ThisThread::sleep_for(100ms);
            my_irq.fall(sw_irq);
            t.stop();
            //------------ IMPORTANT-------------------
            // 1) removes the receive interrupt handler
            pc.enable_input(false);
            // 2) and releases the deep sleep lock
            sleep_manager_can_deep_sleep();
            // 3) then enter Deep Sleep mode
#if   MODE == 1
            ThisThread::sleep_for(10s);
#elif MODE == 2
            thread_sleep_for(10000);
#elif MODE == 3
            wait_ms(10000);
#endif
            system_reset();
        }
        ain = a_in.read();
        myled = !myled;
        seconds = time(NULL);
        strftime(buf, 50, "%H:%M:%S -> ", localtime(&seconds));
        printf("%s", buf);
        printf(
            "analog = %4.3f, processing time=%3d, counter=%4d\r\n",
            ain, t_pass, loop_count++
        );
        t_pass = chrono::duration_cast<chrono::milliseconds>(
                               t.elapsed_time()).count();
        ThisThread::sleep_for(chrono::milliseconds(1000 - t_pass));
        if (pc.readable()) {
            strftime(buf, 50, " %B %d,'%y, %H:%M:%S\r\n", localtime(&seconds));
            printf("[Time] %s\r\n", buf);
            time_enter_mode();
        }
    }
}


void sw_irq(void)
{
    system_reset();
}

void LowPowerConfiguration(void)
{
#if defined(TARGET_NUCLEO_L152RE)
    RCC->AHBENR |=
        (RCC_AHBENR_GPIOAEN | RCC_AHBENR_GPIOBEN | RCC_AHBENR_GPIOCEN |
         RCC_AHBENR_GPIODEN | RCC_AHBENR_GPIOHEN);
#elif defined(TARGET_NUCLEO_L476RG)

#elif defined(TARGET_NUCLEO_F446RE)
    RCC->AHB1ENR |=
        (RCC_AHB1ENR_GPIOAEN | RCC_AHB1ENR_GPIOBEN | RCC_AHB1ENR_GPIOCEN |
         RCC_AHB1ENR_GPIODEN | RCC_AHB1ENR_GPIOHEN);
#endif
    GPIO_InitTypeDef GPIO_InitStruct;
    // All other ports are analog input mode
    GPIO_InitStruct.Pin = GPIO_PIN_All;
    GPIO_InitStruct.Mode = GPIO_MODE_ANALOG;
    GPIO_InitStruct.Pull = GPIO_NOPULL;
    GPIO_InitStruct.Speed = GPIO_SPEED_LOW;
    HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);
    HAL_GPIO_Init(GPIOB, &GPIO_InitStruct);
    HAL_GPIO_Init(GPIOC, &GPIO_InitStruct);
    HAL_GPIO_Init(GPIOD, &GPIO_InitStruct);
    HAL_GPIO_Init(GPIOE, &GPIO_InitStruct);
    HAL_GPIO_Init(GPIOH, &GPIO_InitStruct);
#if defined(TARGET_NUCLEO_L152RE)
    RCC->AHBENR &=
        ~(RCC_AHBENR_GPIOAEN | RCC_AHBENR_GPIOBEN |RCC_AHBENR_GPIOCEN |
          RCC_AHBENR_GPIODEN | RCC_AHBENR_GPIOHEN);
#elif defined(TARGET_NUCLEO_L476RG)
    RCC->AHB1ENR = 0;
    RCC->AHB2ENR = 0;
    RCC->AHB3ENR = 0;
    RCC->APB1ENR2 = 0;
    RCC->APB2ENR = 0;
#elif defined(TARGET_NUCLEO_F446RE)
    RCC->AHB1ENR = 0;
    RCC->AHB2ENR = 0;
    RCC->AHB3ENR = 0;
    RCC->APB1ENR = 0x8;     // alive TIM5
    RCC->APB2ENR = 0;
    RCC->AHB1LPENR = 0;
    RCC->AHB2LPENR = 0;
    RCC->APB1LPENR = 0x8;   // alive TIM5
    RCC->APB2LPENR = 0;
#endif
}

void time_enter_mode(void)
{
    char *ptr;
    char linebuf[64];

    if (pc.readable()) {
        pc.read(linebuf, 1);    // dummy read
    }
    puts("\r\nSet time into RTC");
    puts(" e.g. >21 1 12 13 14 15 -> January 12,'21, 13:14:14");
    linebuf[0] = '>';
    pc.write(linebuf, 1);
    ptr = linebuf;
    get_line(ptr, sizeof(linebuf));
    puts("\r");
    chk_and_set_time(ptr);
}

//  Change string -> integer
int32_t xatoi(char **str, int32_t *res)
{
    uint32_t val;
    uint8_t c, radix, s = 0;

    while ((c = **str) == ' ') (*str)++;
    if (c == '-') {
        s = 1;
        c = *(++(*str));
    }
    if (c == '0') {
        c = *(++(*str));
        if (c <= ' ') {
            *res = 0;
            return 1;
        }
        if (c == 'x') {
            radix = 16;
            c = *(++(*str));
        } else {
            if (c == 'b') {
                radix = 2;
                c = *(++(*str));
            } else {
                if ((c >= '0')&&(c <= '9')) {
                    radix = 8;
                }   else {
                    return 0;
                }
            }
        }
    } else {
        if ((c < '1')||(c > '9')) {
            return 0;
        }
        radix = 10;
    }
    val = 0;
    while (c > ' ') {
        if (c >= 'a') c -= 0x20;
        c -= '0';
        if (c >= 17) {
            c -= 7;
            if (c <= 9) return 0;
        }
        if (c >= radix) return 0;
        val = val * radix + c;
        c = *(++(*str));
    }
    if (s) val = -val;
    *res = val;
    return 1;
}

//  Get key input data
void get_line(char *buff, int32_t len)
{
    char c;
    char bf[8];
    int32_t idx = 0;

    for (;;) {
        pc.read(bf, 1);
        c = bf[0];
        //printf("0x%x \r\n", c);
        if (c == '\r') {
            buff[idx++] = c;
            break;
        }
        if ((c == '\b') && idx) {
            idx--;
            const char bf_bs[] =
            {0x1b, '[', '1', 'D', ' ', 0x1b, '[', '1', 'D'};
            pc.write(bf_bs, 9);
        }
        if (((uint8_t)c >= ' ') && (idx < len - 1)) {
            buff[idx++] = c;
            pc.write(bf, 1);
        }
    }
    buff[idx] = 0;
    bf[0] = '\n';
    pc.write(bf, 1);
}

//  Check key input strings and set time
void chk_and_set_time(char *ptr)
{
    int32_t p1;
    struct tm t;
    time_t seconds;

    if (xatoi(&ptr, &p1)) {
        t.tm_year       = (uint8_t)p1 + 100;
        printf("Year:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_mon        = (uint8_t)p1 - 1;
        printf("Month:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_mday       = (uint8_t)p1;
        printf("Day:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_hour       = (uint8_t)p1;
        printf("Hour:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_min        = (uint8_t)p1;
        printf("Min:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_sec        = (uint8_t)p1;
        printf("Sec: %d \r\n",p1);
    } else {
        return;
    }
    seconds = mktime(&t);
    set_time(seconds);
    // Show Time with several example
    // ex.1
    printf(
        "Date: %04d/%02d/%02d, %02d:%02d:%02d\r\n",
        t.tm_year + 1900, t.tm_mon + 1, t.tm_mday, t.tm_hour, t.tm_min, t.tm_sec
    );
    char buf[64];
    // ex.2
    strftime(buf, 40, "%x %X", localtime(&seconds));
    printf("Date: %s\r\n", buf);
    // ex.3
    strftime(buf, 40, "%I:%M:%S %p (%Y/%m/%d)", localtime(&seconds));
    printf("Date: %s\r\n", buf);
    // ex.4
    strftime(buf, 40, "%B %d,'%y, %H:%M:%S", localtime(&seconds));
    printf("Date: %s\r\n", buf);
}
