/*
 * mbed Library program
 *      Check RTC function and set proper clock if we can set
 *      ONLY FOR "Nucleo Board"
 *
 *  Copyright (c) 2010-2015 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created:  October   24th, 2014
 *      Revised:  May       16th, 2015
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

//#define DEBUG         // use Communication with PC(UART)

//  Include ---------------------------------------------------------------------------------------
#include "mbed.h"
#include "CheckRTC.h"

//  Definition ------------------------------------------------------------------------------------
#ifdef DEBUG
#define PUTS(x)         pcm.puts(x)
#define PRINTF(...)     pcm.printf(__VA_ARGS__)
#else
#define PUTS(x)         {;}
#define PRINTF(...)     {;}
#endif

//  Object ----------------------------------------------------------------------------------------
#ifdef DEBUG
Serial pcm(USBTX, USBRX);
#endif

//  RAM -------------------------------------------------------------------------------------------

//  ROM / Constant data ---------------------------------------------------------------------------

//  Function prototypes ---------------------------------------------------------------------------
#if defined(TARGET_NUCLEO_F401RE) || defined(TARGET_NUCLEO_F411RE) || defined(TARGET_NUCLEO_L152RE)
static int32_t Set_RTC_LSI(void);
static int32_t rtc_external_osc_init(void);
static int32_t Set_RTC_LSE(void);
#endif

//-------------------------------------------------------------------------------------------------
//  Control Program
//-------------------------------------------------------------------------------------------------
int32_t CheckRTC(void)
{
#if defined(TARGET_NUCLEO_F401RE) || defined(TARGET_NUCLEO_F411RE) || defined(TARGET_NUCLEO_L152RE)
    if (rtc_external_osc_init() == OK) {
        return OK;
    } else {
        return NG;
    }
#elif defined(TARGET_LPC1768) || defined(TARGET_K64F)
    return OK;
#else
    return UNKNOWN;
#endif
}

uint32_t get_RTCSEL(void)
{
#if defined(TARGET_NUCLEO_F401RE) || defined(TARGET_NUCLEO_F411RE)
    return ((RCC->BDCR >> 8) & 0x03);
#elif defined(TARGET_NUCLEO_L152RE)
    return ((RCC->CSR >> 16) & 0x03);
#else
    return 0;
#endif
}

#if defined(TARGET_NUCLEO_F401RE) || defined(TARGET_NUCLEO_F411RE) || defined(TARGET_NUCLEO_L152RE)
int32_t Set_RTC_LSE(void)
{
    uint32_t timeout = 0;

    //---------------------------- LSE Configuration -------------------------
    // Enable Power Clock
    __PWR_CLK_ENABLE();
    // Enable write access to Backup domain
    PWR->CR |= PWR_CR_DBP;
    // Wait for Backup domain Write protection disable
    timeout = HAL_GetTick() + DBP_TIMEOUT_VALUE;
    while((PWR->CR & PWR_CR_DBP) == RESET) {
        if(HAL_GetTick() >= timeout) {
            PRINTF("Time-Out 1\r\n");
            return NG;
        }
    }
    // Reset LSEON and LSEBYP bits before configuring the LSE ----------------
    __HAL_RCC_LSE_CONFIG(RCC_LSE_OFF);
    // Get timeout
    timeout = HAL_GetTick() + TIMEOUT;
    // Wait till LSE is ready
    while(__HAL_RCC_GET_FLAG(RCC_FLAG_LSERDY) != RESET) {
        if(HAL_GetTick() >= timeout) {
            PRINTF("Time-Out 2\r\n");
            return NG;
        }
    }
    // Set the new LSE configuration -----------------------------------------
    __HAL_RCC_LSE_CONFIG(RCC_LSE_ON);
    // Get timeout
    timeout = HAL_GetTick() + TIMEOUT;
    // Wait till LSE is ready
    while(__HAL_RCC_GET_FLAG(RCC_FLAG_LSERDY) == RESET) {
        if(HAL_GetTick() >= timeout) {
            PRINTF("Time-Out 3\r\n");
            return NG;
        }
    }
    PRINTF("OK");
    return OK;
}

int32_t Set_RTC_LSI(void)
{
    uint32_t timeout = 0;

    // Enable Power clock
    __PWR_CLK_ENABLE();
    // Enable access to Backup domain
    HAL_PWR_EnableBkUpAccess();
    // Reset Backup domain
    __HAL_RCC_BACKUPRESET_FORCE();
    __HAL_RCC_BACKUPRESET_RELEASE();
    // Enable Power Clock
    __PWR_CLK_ENABLE();
    // Enable write access to Backup domain
    PWR->CR |= PWR_CR_DBP;
    // Wait for Backup domain Write protection disable
    timeout = HAL_GetTick() + DBP_TIMEOUT_VALUE;
    while((PWR->CR & PWR_CR_DBP) == RESET) {
        if(HAL_GetTick() >= timeout) {
            return NG;
        }
    }
    __HAL_RCC_LSE_CONFIG(RCC_LSE_OFF);
    // Enable LSI
    __HAL_RCC_LSI_ENABLE();
    timeout = HAL_GetTick() + TIMEOUT;
    // Wait till LSI is ready
    while(__HAL_RCC_GET_FLAG(RCC_FLAG_LSIRDY) == RESET) {
        if(HAL_GetTick() >= timeout) {
            return NG;
        }
    }
    // Connect LSI to RTC
    __HAL_RCC_RTC_CLKPRESCALER(RCC_RTCCLKSOURCE_LSI);
    __HAL_RCC_RTC_CONFIG(RCC_RTCCLKSOURCE_LSI);
    return OK;
}

int32_t rtc_external_osc_init(void)
{
    // Enable Power clock
    __PWR_CLK_ENABLE();
    // Enable access to Backup domain
    HAL_PWR_EnableBkUpAccess();
    // Reset Backup domain
    __HAL_RCC_BACKUPRESET_FORCE();
    __HAL_RCC_BACKUPRESET_RELEASE();
    // Enable LSE Oscillator
    if (Set_RTC_LSE() == OK) {
        // Connect LSE to RTC
        __HAL_RCC_RTC_CLKPRESCALER(RCC_RTCCLKSOURCE_LSE);
        __HAL_RCC_RTC_CONFIG(RCC_RTCCLKSOURCE_LSE);
        return OK;
    } else {
        Set_RTC_LSI();
        return NG;
    }
}
#endif
