/*
 * Mbed library program
 *  Control CAT24M01 EEPROM
 *
 * Copyright (c) 2017 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created: September  19th, 2017
 *      Revised: September  19th, 2017
 */

#ifndef CAT24M01_H
#define CAT24M01_H

#include "mbed.h"

/** Interface for 1Mbit Serial EEPROM (I2C Interface)
 *    ON Semiconductor. CAT24M01
 *
 * @code
 * #include "mbed.h"
 * #include "CAT24M01.h"
 *
 * // I2C Communication
 * CAT24M01   eep(I2C_SDA, I2C_SCL);   // SDA, SCL
 * // If you connected I2C line not only this device but also other devices,
 * //     you need to declare following method.
 * I2C        i2c(I2C_SDA, I2C_SCL);   // SDA, SCL
 * CAT24M01   eep(i2c);                // ON Semiconductor 1Mbit EE-PROM
 *
 * int main() {
 *   eep.write(0,dt);     // write addr=0 data=dt
 *   dt = eep.read(0);    // read addr=0
 *   // page write mode, addr top=0x1ff00, 256bytes data (save it in eep_buf[])
 *   status = eep.write_page(0x1ff00, eep_buf, sizeof(eep_buf));
 * }
 * @endcode
 */

// EEPROM ON Semiconductor, CAT24M01
// Address b7=1,b6=0,b5=1,b4=0,b3=A2pin,b2=A1pin,b1=page addr, b0=R/W
#define ICHIGOROMADDR   0xa0        //  as initial data (A2=A1=GND)
#define CAT24M01ADDR0   0xa0        //  as initial data (A2=A1=GND)
#define CAT24M01ADDR1   0xa4        //  as initial data (A2=GND, A1=VCC)
#define CAT24M01ADDR2   0xa8        //  as initial data (A2=VCC, A1=GND)
#define CAT24M01ADDR3   0xac        //  as initial data (A2=VCC, A1=VCC)

typedef enum {
    CAT24_OK = 0,
    CAT24_BUSY,
    CAT24_WRONG_BF_SIZE,
    CAT24_WRONG_TOP_ADDR,
    CAT24_ERROR
} CAT24_STATUS;

class CAT24M01
{
public:
    /** Configure data pin
      * @param data SDA and SCL pins
      */
    CAT24M01(PinName p_sda, PinName p_scl, uint8_t addr = CAT24M01ADDR0);

    /** Configure data pin (with other devices on I2C line)
      * @param I2C previous definition
      */
    CAT24M01(I2C& p_i2c, uint8_t addr = CAT24M01ADDR0);

    /** Read one data from EEPROM
      * @param dat_address (0 to 0x1ffff)
      * @return read data
      */
    uint8_t read(uint32_t addr);

    /** Write one data to EEPROM
      * @param data_address (o to 0x1ffff), data (byte)
      * @return none
      */
    void write(uint32_t addr, uint8_t dt);

    /** Page read (256 bytes) from EEPROM
      * @param top_address (o to 0x1ffxx) e.g. 0x00,0x100, --- 0x1fe00, 0x1ff00
      * @param dt -> buffer(need to keep 258 bytes), data top = buffer[0]
      * @param size -> sizeof(buffer)
      * @return status 0=OK, others=Error
      */
    CAT24_STATUS read_page(uint32_t addr_page_top, uint8_t *dt, int size);

    /** Page write (256 bytes) to EEPROM
      * @param top_address (o to 0x1ffxx) e.g. 0x00,0x100, --- 0x1fe00, 0x1ff00
      * @param dt -> buffer(need to keep 258 bytes), data top = buffer[0]
      * @param size -> sizeof(buffer)
      * @return status 0=OK, others=Error
      */
    CAT24_STATUS write_page(uint32_t addr_page_top, uint8_t *dt, int size);

    /** Set I2C clock frequency
      * @param freq.
      * @return none
      */
    void frequency(int hz);

protected:
    I2C *_i2c_p;
    I2C &_i2c;

private:
    uint8_t CAT24M01_addr;
};

#endif // CAT24M01_H
