/*
 * Mbed library program
 *  Control CAT24M01 EEPROM
 *
 * Copyright (c) 2017 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created: September  19th, 2017
 *      Revised: September  19th, 2017
 */

#include "mbed.h"
#include "CAT24M01.h"

CAT24M01::CAT24M01 (PinName p_sda, PinName p_scl, uint8_t addr)
    : _i2c_p(new I2C(p_sda, p_scl)), _i2c(*_i2c_p)
{
    CAT24M01_addr = addr;
}

CAT24M01::CAT24M01 (I2C& p_i2c, uint8_t addr)
    : _i2c(p_i2c)
{
    CAT24M01_addr = addr;
}

uint8_t CAT24M01::read(uint32_t addr)
{
    uint8_t eep_dt[2];
    uint8_t tmp_addr;

    tmp_addr = CAT24M01_addr;
    if ( addr & 0x10000 ) { // addr > 0xffff then P0 sets 1
        tmp_addr += 0x02;       // P0=1
    }
    eep_dt[0] = ( uint8_t )( (addr & 0xff00) >> 8 );
    eep_dt[1] = ( uint8_t )( (addr & 0xff) );
    _i2c.write((uint8_t)tmp_addr, (char *)eep_dt, 2);
    _i2c.read((uint8_t)tmp_addr, (char *)eep_dt, 1);
    return eep_dt[0];
}

void CAT24M01::write(uint32_t addr, uint8_t dt)
{
    uint8_t eep_dt[3];
    uint8_t tmp_addr;

    tmp_addr = CAT24M01_addr;
    if ( addr & 0x10000 ) { // addr > 0xffff then P0 sets 1
        tmp_addr += 0x02;       // P0=1
    }
    eep_dt[0] = ( uint8_t )( (addr & 0xff00) >> 8 );
    eep_dt[1] = ( uint8_t )( (addr & 0xff) );
    eep_dt[2] = dt;
    _i2c.write((uint8_t)tmp_addr, (char *)eep_dt, 3);
}

CAT24_STATUS CAT24M01::read_page(uint32_t addr_page_top, uint8_t *dt, int size)
{
    uint8_t eep_dt[2];
    uint8_t tmp_addr;

    if (size >= 256 + 2) {
        if ((addr_page_top & 0xff) == 0) {
            tmp_addr = CAT24M01_addr;
            if ( addr_page_top & 0x10000 ) { // addr > 0xffff then P0 sets 1
                tmp_addr += 0x02;               // P0=1
            }
            eep_dt[0] = ( uint8_t )( (addr_page_top & 0xff00) >> 8 );
            eep_dt[1] = 0;
            _i2c.write((uint8_t)tmp_addr, (char *)eep_dt, 2);
            _i2c.read((uint8_t)tmp_addr, (char *)dt, 256);
            return CAT24_OK;
        } else {
            return CAT24_WRONG_TOP_ADDR;
        }
    } else {
        return CAT24_WRONG_BF_SIZE;
    }
}

CAT24_STATUS CAT24M01::write_page(uint32_t addr_page_top, uint8_t *dt, int size)
{
    uint8_t tmp_addr;
    int i;

    if (size >= 256 + 2) {
        if ((addr_page_top & 0xff) == 0) {
            for (i = 255 + 2; i > 1; i--) { // shift data
                dt[i] = dt[i-2];
            }
            tmp_addr = CAT24M01_addr;
            if ( addr_page_top & 0x10000 ) { // addr > 0xffff then P0 sets 1
                tmp_addr += 0x02;               // P0=1
            }
            dt[0] = ( uint8_t )( (addr_page_top & 0xff00) >> 8 );
            dt[1] = 0;
            _i2c.write((uint8_t)tmp_addr, (char *)dt, 256 + 2);
            return CAT24_OK;
        } else {
            return CAT24_WRONG_TOP_ADDR;
        }
    } else {
        return CAT24_WRONG_BF_SIZE;
    }
}

void CAT24M01::frequency(int hz)
{
    _i2c.frequency(hz);
}

