/*
 * mbed library program
 *  LCD Module: 40 characters × 4 lines, F-STN / Black&White-mode
 *  C-51849NFJ-SLW-ADN by KYOCERA Industrial Ceramics Corporation - Display Division.
 *
 * Copyright (c) 2015 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created: April     27th, 2015
 *      Revised: April     28th, 2015
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
/*
 *---------------- REFERENCE ----------------------------------------------------------------------
 *  http://www.kyocera-display.com/products/partdetail.asp?PartNumber=C-51849NFJ-SLW-ADN
 *  http://akizukidenshi.com/catalog/g/gP-05664/
 */

#ifndef C51849NFJ_H
#define C51849NFJ_H

#include "mbed.h"

/**  A Text LCD (40x4) interface for C-51849NFJ-SLW-ADN by KYOCERA (OPTREX)
 *
 * @code
 * #include "mbed.h"
 * #include "C51849NFJ.h"
 *
 * C51849NFJ lcd(D7, D8, D9, D10, D11, D12, D13); // rs, e1, e2, d4-d7
 *
 * int main() {
 *     lcd.printf("Hello World!\n");
 * }
 * @endcode
 */

class C51849NFJ : public Stream {
public:

    /** Create a TextLCD interface
     * @param rs    Instruction/data control line
     * @param e1,e2 Enable line (clock) for two chips NT3881 controler
     * @param d4-d7 Data lines for using as a 4-bit interface
     */
    C51849NFJ(PinName rs, PinName e1, PinName e2, PinName d4, PinName d5, PinName d6, PinName d7);

#if DOXYGEN_ONLY
    /** Write a character to the LCD
     * @param c The character to write to the display
     */
    int putc(int c);

    /** Write a formated string to the LCD
     * @param format A printf-style format string
     */
    int printf(const char* format, ...);
#endif

    /** Locate to a screen column and row
     * @param column  horizontal position from the left, indexed from 0
     * @param row     vertical position from the top, indexed from 0
     */
    void locate(int column, int row);

    /** Clear the screen and locate to 0,0 */
    void cls();

protected:
    virtual int _putc(int value);
    virtual int _getc();
    int16_t address(int column, int row);
    void character(int column, int row, int c);
    void writeCMD(int command);

    DigitalOut _rs, _e1, _e2;
    BusOut _d;

    int _column;
    int _row;
};

#endif      // C51849NFJ_H
