/**
 * @file    BufferedSoftSerial.cpp
 * @brief   Software Buffer
 *          - Extends mbed Serial functionallity adding irq driven TX and RX
 * @author  sam grove
 * @version 1.0
 * @see
 *
 * Copyright (c) 2013
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// Modified by K.Arai / JH1PJL     May 15th, 2020

#include "Bf_SoftSerial_IR.h"
#include <stdarg.h>

Bf_SoftSerial_IR::Bf_SoftSerial_IR(PinName tx, PinName rx, const char* name)
    : SoftSerial_IR(tx, rx, name)
{
    _rxbuf.reset();
    _txbuf.reset();
    SoftSerial_IR::attach(
        this, &Bf_SoftSerial_IR::rx_Irq, SoftSerial_IR::RxIrq
    );
    return;
}

int Bf_SoftSerial_IR::readable(void)
{
    return !_rxbuf.empty();
}

int Bf_SoftSerial_IR::writeable(void)
{
    return _txbuf.full();
}

int Bf_SoftSerial_IR::getc(void)
{
    char c;
    while(_rxbuf.empty()) {
        ;
    }
    _rxbuf.pop(c);
    return c;
}

int Bf_SoftSerial_IR::putc(int c)
{
    _txbuf.push((char)c);
    prime();
    return c;
}

int Bf_SoftSerial_IR::puts(const char *s)
{
    const char* ptr = s;

    while(*(ptr) != 0) {
        _txbuf.push(*(ptr++));
    }
#if 0
    _txbuf.push('\n');  // done per puts definition
#endif
    prime();
    return (ptr - s) + 1;
}

int Bf_SoftSerial_IR::printf(const char* format, ...)
{
    char buf[256] = {0};
    int r = 0;

    va_list arg;
    va_start(arg, format);
    r = vsprintf(buf, format, arg);
    // this may not hit the heap but should alert the user anyways
    if(r > sizeof(buf)) {
        error("%s %d buffer overwrite!\n", __FILE__, __LINE__);
    }
    va_end(arg);
    r = write(buf, r);
    return r;
}

ssize_t Bf_SoftSerial_IR::write(const void *s, size_t length)
{
    const char* ptr = (const char*)s;
    const char* end = ptr + length;

    while (ptr != end) {
        _txbuf.push(*(ptr++));
    }
    prime();
    return ptr - (const char*)s;
}


void Bf_SoftSerial_IR::rx_Irq(void)
{
    // read from the peripheral and make sure something is available
    if(SoftSerial_IR::readable()) {
        _rxbuf.push(_getc()); // if so load them into a buffer
    }
    return;
}

void Bf_SoftSerial_IR::tx_Irq(void)
{
    // see if there is room in the hardware fifo and
    //  if something is in the software fifo
    while(SoftSerial_IR::writeable()) {
        if(!_txbuf.empty()) {
            char c;
            _txbuf.pop(c);
            _putc((int)c);
        } else {
            // disable the TX interrupt when there is nothing left to send
            SoftSerial_IR::attach(NULL, SoftSerial_IR::TxIrq);
            break;
        }
    }
    return;
}

void Bf_SoftSerial_IR::prime(void)
{
    // if already busy then the irq will pick this up
    if(SoftSerial_IR::writeable()) {
        // make sure not to cause contention in the irq
        SoftSerial_IR::attach(NULL, SoftSerial_IR::TxIrq);
        // only write to hardware in one place
        tx_Irq();
        SoftSerial_IR::attach(
            this,&Bf_SoftSerial_IR::tx_Irq,SoftSerial_IR::TxIrq
        );
    }
    return;
}
