/*
 * mbed library program 
 *  Control BMP180(Bosch) Pressure Sensor
 *
 * Copyright (c) 2013,'14,'17 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created: August     14th, 2013   for STM32L152 
 *      Changed: May        21st, 2014   mbed LPC1114
 *      Revised: August     23rd, 2017
 */

#ifndef BMP180_H
#define BMP180_H

#include "mbed.h"

//  Bosch barmeter ID
#define BMP180_CHIP_ID          0x55

/** Interface for Bosch Pressure Sensor (I2C Interface) BMP180
 *
 * Measurement Air pressure (Barometer) and temperature via I2C interface.
 *
 *  Chip has compensation data in the sensor (inside of EEPROM).
 *
 *  Normalization is specified in the documentation as follows.
 *
 *      Bosch Sensortec BMP180 Datasheet : BST-BMP180-DS000-09 Revision: 2.5  Date: 5 April 2013
 *
 * @code
 * #include "mbed.h"
 *
 * // I2C Communication
 *  BMP180(PinName p_sda, PinName p_scl);   // SDA, SCL
 * // If you connected I2C line not only this device but also other devices,
 * //     you need to declare following method.
 *  I2C         i2c(dp5,dp27);              // SDA, SCL
 *  BMP180(I2C& p_i2c);
 *
 * int main() {
 * float pressure, temperature;
 *
 *   bmp180.normalize();    // This is important function Data read from BMP180 then normalization
 *   pressure = bmp180.read_pressure();         // just read the data
 *   temperature = bmp180.read_temperature();   // just read the data
 * }
 * @endcode
 */

class BMP180 {
public:
    /** Configure data pin
      * @param data SDA and SCL pins
      */
    BMP180(PinName p_sda, PinName p_scl);
    
    /** Configure data pin (with other devices on I2C line)
      * @param I2C previous definition
      */
    BMP180(I2C& p_i2c);

    /** Read a float type data from BMP180
      * @param none
      * @return temperature unit:degreeC(Celsius)
      */
    float read_temperature();
    
    /** Read a float type data from BMP180
      * @param none
      * @return pressure unit:hPa(hectopascals)
      */
    float read_pressure();
    
    /** Read a BMP180 ID number
      * @param none
      * @return if BMP180, it should be 0x55.
      */
    uint8_t read_baro_id();

    /** Read press and temp data from BMP180 then normalize the data.
      * @param none
      * @return none (The result is able to read read_temperature() or read_pressure()).
      */
    void normalize();

protected:
    void init(void);
    void _i2c_read_n_bytes(int, char*, int);
    void _i2c_write_n_bytes(int, char*, int);

    I2C *_i2c_p;
    I2C &_i2c;

    float temperature;
    float pressure;
    uint8_t id_number;

private:
    //  save EEPROM Data (Coefficient data)
    int16_t  eep_ac1, eep_ac2, eep_ac3;
    uint16_t eep_ac4, eep_ac5, eep_ac6;
    int16_t  eep_b1,  eep_b2,  eep_mb;     // eep_mb:not use
    int16_t  eep_mc,  eep_md;
    // temporary save
    char     baro_dt[4];
    //  address
    char BMP180_addr;
};

#endif // BMP180_H

