/* mbed Microcontroller Library
 * Copyright (c) 2006-2014 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 *  Modified by Kenji Arai
 *      http://www.page.sannet.ne.jp/kenjia/index.html
 *      http://mbed.org/users/kenjiArai/
 *
 *      Started:  January    3rd, 2016
 *      Revised:  January    8th, 2017
 */

//  Include ---------------------------------------------------------------------------------------
#include "mbed.h"
#include "BLE.h"
#include "BatteryService.h"
#include "nRF51_Vdd.h"

//  Definition ------------------------------------------------------------------------------------
#define NEED_CONSOLE_OUTPUT     1 /* Set this if you need debug messages on the console;
                                   * it will have an impact on code-size and power consumption. */

#if NEED_CONSOLE_OUTPUT
#define DEBUG(...) { printf(__VA_ARGS__); }
#else
#define DEBUG(...) /* nothing */
#endif /* #if NEED_CONSOLE_OUTPUT */

//  Object ----------------------------------------------------------------------------------------
BLE         ble;
DigitalOut  led1(LED1, 1);
//Serial      pc(USBTX, USBRX);
Serial      pc(P0_3, P0_1);     // TX, RX
I2C         i2c(P0_29,P0_28);
SPI         spi(P0_20,P0_22,P0_25);
nRF51_Vdd   vdd(3.6f, 2.6f);
Ticker      t;

//  ROM / Constant data ---------------------------------------------------------------------------
const char *deviceName = "mbedMon";

//  RAM -------------------------------------------------------------------------------------------
BatteryService *batteryService = NULL;
uint8_t batteryLevel = 50;

//  Function prototypes ---------------------------------------------------------------------------
extern void debug_interface(uint8_t mode);

//-------------------------------------------------------------------------------------------------
//  Control Program
//-------------------------------------------------------------------------------------------------
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *disconnectionParams)
{
    DEBUG("Disconnected handle %u!\n\r", disconnectionParams->handle);
    DEBUG("Restarting the advertising process\n\r");
    BLE::Instance(BLE::DEFAULT_INSTANCE).gap().startAdvertising(); // restart advertising
}

void blink(void)
{
    led1 = !led1;
}

int main(void)
{
    // ******************************************************
    // Here is a example to impliment the monitor
    // para: 1 -> goto montor and never comeback
    // ******************************************************
    debug_interface(1);

    // Application program runs at here
    t.attach(blink, 1.0f);

    DEBUG("Initialising the nRF51822\r\n");
    ble.init();
    ble.setDeviceName((const uint8_t *)deviceName);
    ble.onDisconnection(disconnectionCallback);

    batteryService = new BatteryService(ble, batteryLevel);

    /* setup advertising */
    ble.accumulateAdvertisingPayload
     (GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME,
                                    (const uint8_t *)deviceName,
                                     strlen(deviceName));
    ble.setAdvertisingInterval(1000); /* 1000ms; in multiples of 0.625ms. */
    ble.startAdvertising();

    while (true) {
        // this will return upon any system event (such as an interrupt or a ticker wakeup)
        ble.waitForEvent();

        // the magic battery processing
        batteryLevel++;
#if 1
        if (batteryLevel > 100) {
            batteryLevel = 20;
        }
        DEBUG("Vdd: %4.3f\r\n", vdd.read_real_value());
#else
        DEBUG("Vdd: %f\r\n", vdd.read_real_value());
        batteryLevel = vdd.read();
#endif

        batteryService->updateBatteryLevel(batteryLevel);
        DEBUG("battery=%d\r\n", batteryLevel);

        // ******************************************************
        // Here is a example to impliment the monitor
        // para: 0 -> if Uart RX Data is ready then goto montor
        // ******************************************************
        debug_interface(0);
    }
}
