/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 *      January   6th, 2016     Modified by Kenji Arai
 *      Feburary 19th, 2016
 *                              http://www.page.sannet.ne.jp/kenjia/index.html
 *                              http://mbed.org/users/kenjiArai/
 *
 *      Original:
 *          1) BLE_UARTConsole
 *             https://developer.mbed.org/teams/Bluetooth-Low-Energy/code/BLE_UARTConsole/
 *          2) BLE_HTM_by_InTempSensr_pub
 *             https://developer.mbed.org/users/takafuminaka/code/BLE_HTM_by_InTempSensr_pub/
 *             Tanks Takafumi Naka san!
 *      Tested Controller Device:
 *          iPhone6 UART application in nRF Toolbox
 *          https://itunes.apple.com/jp/app/nrf-toolbox/id820906058?mt=8
 *          Please check data on "Show log" screen
 */

//  Include ---------------------------------------------------------------------------------------
#include <string.h>
#include "mbed.h"
#include "BLE.h"
#include "UARTService.h"
#include "nrf_soc.h"            // for internal Thermo sensoer
#include "nRF51_Vdd.h"          // Read nRF51 Vdd voltage

//  Definition ------------------------------------------------------------------------------------
#define NEED_CONSOLE_OUTPUT 1   // Set this if you need debug messages on the console;

#if NEED_CONSOLE_OUTPUT
#define DEBUG(...) { pc.printf(__VA_ARGS__); }
#else
#define DEBUG(...) /* nothing */
#endif

//  Object ----------------------------------------------------------------------------------------
BLE         ble;
UARTService *uart;
nRF51_Vdd   vdd(3.0f, 2.2f);    // CR2032 3V-2.2V (http://rbs.ta36.com/?p=20398#CR2032)
#if NEED_CONSOLE_OUTPUT
Serial      pc(USBTX,USBRX);
#endif

//  RAM -------------------------------------------------------------------------------------------
static volatile bool triggerSensorPolling = false; 

//  ROM / Constant data ---------------------------------------------------------------------------
const char *deviceName = "mbedTY";

//  Function prototypes ---------------------------------------------------------------------------
void Update_Values(void);

//-------------------------------------------------------------------------------------------------
//  Control Program
//-------------------------------------------------------------------------------------------------
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params){
    if (uart){
        uart->writeString("Disconnected!\r\n");
        uart->writeString("Restarting the advertising process\r\n");
    }
    ble.startAdvertising();
}

void periodicCallback(void){
    triggerSensorPolling = true;
}

int main(void){
    // Set priodic interrupt
    Ticker ticker;
    ticker.attach(periodicCallback, 5);
    // Opening screen
    DEBUG("\r\nInitialising the nRF51822\r\n");
    DEBUG("Temperature and CPU Vdd voltage via UART service\r\n");
    // setup BLE
    ble.init();
    ble.setDeviceName((const uint8_t *)deviceName);
    ble.onDisconnection(disconnectionCallback);
    // BLE Uart
    uart = new UARTService(ble);
    // setup advertising
    ble.gap().setTxPower(-40);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME,
                                     (const uint8_t *)deviceName,
                                     strlen(deviceName));
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
                                     (const uint8_t *)UARTServiceUUID_reversed,
                                      sizeof(UARTServiceUUID_reversed));
    ble.setAdvertisingInterval(160); /* 100ms; in multiples of 0.625ms. */
    ble.startAdvertising();
    while (true){
        if (triggerSensorPolling){
            triggerSensorPolling = false;
            Update_Values();           
        } else {
            ble.waitForEvent();
        }
    }
}

void fill_space(char *bf){
    uint8_t n;

    n = strlen(bf);
    bf += n;
    for (uint8_t i = n; i <= 18; bf++, i++){
        *bf = ' ';
    }
    *bf = '.';
    *(bf + 1) = 0;
}

void Update_Values(void){
    char buf[32];
    int32_t p_temp;
    float temperature;

    // Update a temperature (inside nRF51822 chip)
    sd_temp_get(&p_temp);
    // -16.0f is offset vale for chip die temp to ambient temp (depend on your board)
    temperature = float(p_temp) / 4; // Original = float(p_temp)/4.0f - 16.0f;
    //           12      89
    sprintf(buf,"T:%+4.1fdegC", temperature);
    fill_space(buf);
    if (uart){
        uart->writeString(buf);
    }
    DEBUG("%s\b\r\n", buf);
    // Update a Vdd voltage
    //           01234     9
    sprintf(buf,"Vdd:%3.2fV", vdd.read_real_value());
    fill_space(buf);
    if (uart){
        uart->writeString(buf);
    }
    DEBUG("%s\b\r\n", buf);
    //           123456789 0
    sprintf(buf,"Charge:%d%%", vdd.read());
    fill_space(buf);
    if (uart){
        uart->writeString(buf);
    }
    DEBUG("%s\b\r\n", buf);
}
