/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 *      January   6th, 2016     Modified by Kenji Arai
 *      August    9th, 2018     TY51822r3 runs on mbed-os-5.9.4
 *
 *                              http://www.page.sannet.ne.jp/kenjia/index.html
 *                              http://mbed.org/users/kenjiArai/
 *  Original:
 *  1) BLE_UARTConsole
 *      https://developer.mbed.org/teams/
                    Bluetooth-Low-Energy/code/BLE_UARTConsole/
 *  2) BLE_HTM_by_InTempSensr_pub
 *      https://developer.mbed.org/users/
                    takafuminaka/code/BLE_HTM_by_InTempSensr_pub/
 *      Tanks Takafumi Naka san!
 *  Tested Controller Device:
 *          iPhone6 UART application in nRF Toolbox
 *          https://itunes.apple.com/jp/app/nrf-toolbox/id820906058?mt=8
 *          Please check data on "Show log" screen
 */

//  Include --------------------------------------------------------------------
#include <string.h>
#include "mbed.h"
#include "BLE.h"
#include "UARTService.h"
#include "nrf_soc.h"            // for internal Thermo sensoer
#include "nRF51_Vdd.h"          // Read nRF51 Vdd voltage

//  Definition -----------------------------------------------------------------

//  Object ---------------------------------------------------------------------
BLE&        ble_x = BLE::Instance();
UARTService *uart;
nRF51_Vdd   vdd(3.6f, 2.6f);

//  RAM ------------------------------------------------------------------------
static volatile bool triggerSensorPolling = false; 

//  ROM / Constant data --------------------------------------------------------
const char *deviceName = "mbedBLE";

//  Function prototypes --------------------------------------------------------
void Update_Values(void);

//------------------------------------------------------------------------------
//  Control Program
//------------------------------------------------------------------------------
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params){
    if (uart){
        uart->writeString("Disconnected!\r\n");
        uart->writeString("Restarting the advertising process\r\n");
    }
    ble_x.startAdvertising();
}

void periodicCallback(void){
    triggerSensorPolling = true;
}

int main(void){
    // Set priodic interrupt
    Ticker ticker;
    ticker.attach(periodicCallback, 5);
    // Opening screen
    printf("\r\nInitialising TY51822r3\r\n");
    printf("Temperature and CPU Vdd voltage via UART service\r\n");
    // setup BLE
    ble_x.init();
    ble_x.setDeviceName((const uint8_t *)deviceName);
    ble_x.onDisconnection(disconnectionCallback);
    // BLE Uart
    uart = new UARTService(ble_x);
    // setup advertising
    ble_x.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    ble_x.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble_x.accumulateAdvertisingPayload(
        GapAdvertisingData::SHORTENED_LOCAL_NAME,
        (const uint8_t *)deviceName,
        strlen(deviceName)
    );
    ble_x.accumulateAdvertisingPayload(
        GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
        (const uint8_t *)UARTServiceUUID_reversed,
        sizeof(UARTServiceUUID_reversed)
    );
    ble_x.setAdvertisingInterval(160); /* 100ms; in multiples of 0.625ms. */
    ble_x.startAdvertising();
    while (true){
        if (triggerSensorPolling){
            triggerSensorPolling = false;
            Update_Values();           
        } else {
            ble_x.waitForEvent();
        }
    }
}

void Update_Values(void){
    char buf[32];
    int32_t p_temp;
    float temperature;

    // Update a temperature (inside nRF51822 chip)
    sd_temp_get(&p_temp);
    temperature = float(p_temp) / 4.0f;
    //           12      89
    sprintf(buf,"T:%+4.1fdC", temperature);
    if (uart){
        uart->writeString(buf);
    }
    printf(buf);
    // Update a Vdd voltage
    //           01234     9
    sprintf(buf,",Vdd:%3.2fV  ", vdd.read_real_value());
    if (uart){
        uart->writeString(buf);
    }
    printf(buf);
    //           123456789 0
    sprintf(buf,"Charge:%d%% \r\n", vdd.read());
    if (uart){
        uart->writeString(buf);
    }
    printf(buf);
}
