/*
 *      December 31st, 2015     Modified by Kenji Arai
 *		January  19th, 2016
 *		October  12th, 2017		!! Run on Mbed-os5
 *                              http://www.page.sannet.ne.jp/kenjia/index.html
 *                              http://mbed.org/users/kenjiArai/
 *      -> Works well with latest Rev.
 *
 *      Original:
 *          BLE_RCBController2_Motor
 *          https://developer.mbed.org/users/robo8080/code/BLE_RCBController2_Motor/
 *      Tested Controller Device:
 *          iPhone6 RCBController (Ver1.4.0)
 *          https://itunes.apple.com/jp/app/rcbcontroller/id689724127?mt=8
 */

//  Include ---------------------------------------------------------------------------------------
#include "mbed.h"
#include "ble/BLE.h"
#include "RCBController.h"
#include "Servo.h"

//  Definition ------------------------------------------------------------------------------------
#define NEED_CONSOLE_OUTPUT			1		// Set this if you need debug messages on the console

#if NEED_CONSOLE_OUTPUT
#define DEBUG(...) { printf(__VA_ARGS__); }
#else
#define DEBUG(...)
#endif

#define LED_A_PIN					P0_21
#define LED_B_PIN					P0_22
#define LED_X_PIN					P0_23
#define LED_Y_PIN					P0_24
#define SERVO_0_PIN					P0_18
#define SERVO_1_PIN					P0_16

//  Object ----------------------------------------------------------------------------------------
BLE  			ble;
Servo   		SERVO_0(SERVO_0_PIN);
Servo   		SERVO_1(SERVO_1_PIN);
DigitalOut      LED_A(LED_A_PIN);
DigitalOut      LED_B(LED_B_PIN);
DigitalOut      LED_X(LED_X_PIN);
DigitalOut      LED_Y(LED_Y_PIN);

//  RAM -------------------------------------------------------------------------------------------
uint8_t RCBControllerPayload[10] = {0,};
RCBController 		controller;

//  ROM / Constant data ---------------------------------------------------------------------------
// RCBController Service
static const uint16_t RCBController_service_uuid = 0xFFF0;
static const uint16_t RCBController_Characteristic_uuid = 0xFFF1;
const char *deviceName = "mbedBLE";

//  Function prototypes ---------------------------------------------------------------------------
GattCharacteristic  ControllerChar (RCBController_Characteristic_uuid,RCBControllerPayload,10, 10,
							GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | 
							GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE_WITHOUT_RESPONSE);
GattCharacteristic *ControllerChars[] = {&ControllerChar};
GattService         RCBControllerService(RCBController_service_uuid, ControllerChars,
							sizeof(ControllerChars) / sizeof(GattCharacteristic *));

//-------------------------------------------------------------------------------------------------
//  Control Program
//-------------------------------------------------------------------------------------------------
void data_analysis(void){
	static uint8_t cont_flg_A = 0;
	static uint8_t cont_flg_B = 0;
	static uint8_t cont_flg_X = 0;
	static uint8_t cont_flg_Y = 0;
	float value;

	if (controller.status.A == 1){
		if (cont_flg_A == 0){
			LED_A = !LED_A;
			cont_flg_A = 1;
		}
	} else {
		cont_flg_A = 0;
	}
	if (controller.status.B == 1){
		if (cont_flg_B == 0){
			LED_B = !LED_B;
			cont_flg_B = 1;
		}
	} else {
		cont_flg_B = 0;
	}
	if (controller.status.X == 1){
		if (cont_flg_X == 0){
			LED_X = !LED_X;
			cont_flg_X = 1;
		}
	} else {
		cont_flg_X = 0;
	}
	if (controller.status.Y == 1){
		if (cont_flg_Y == 0){
			LED_Y = !LED_Y;
			cont_flg_Y = 1;
		}
	} else {
		cont_flg_Y = 0;
	}
	value = (float)controller.status.LeftAnalogUD/255 * 150.0f + 20;
	SERVO_0.write(value);
	value = (float)controller.status.LeftAnalogLR/255 * 150.0f + 20;
	SERVO_1.write(value);	
}

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params){
    DEBUG("Disconnected!\r\n");
    DEBUG("Restarting the advertising process\r\n");
    ble.startAdvertising();
}

void onDataWritten(const GattWriteCallbackParams *params){
	if (params->handle == ControllerChar.getValueAttribute().getHandle()) {
		uint16_t bytesRead;
	 	ble.readCharacteristicValue(ControllerChar.getValueAttribute().getHandle(),
	 								RCBControllerPayload, &bytesRead);
        memcpy( &controller.data[0], RCBControllerPayload, sizeof(controller));
		DEBUG("DATA:0x%02x 0x%02x %d %d %d %d %d %d %d 0x%02x\r\n",
				controller.data[0],controller.data[1],controller.data[2],controller.data[3],
				controller.data[4],controller.data[5],controller.data[6],controller.data[7],
				controller.data[8],controller.data[9]);
		data_analysis();	
	}
}

int main(void){
    DEBUG("\r\nInitialising the nRF51822\r\n");
    ble.init();
    ble.setDeviceName((const uint8_t *)deviceName);
    ble.onDisconnection(disconnectionCallback);
    ble.onDataWritten(onDataWritten);
	DEBUG("Start RCB Controller\r\n");
    /* setup advertising */
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME,
                                    (const uint8_t *)deviceName, strlen(deviceName));
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS,
                                    (const uint8_t *)RCBController_service_uuid,
                                     sizeof(RCBController_service_uuid));
    ble.setAdvertisingInterval(160); /* 100ms; in multiples of 0.625ms. */
    ble.startAdvertising();
    ble.addService(RCBControllerService);
    while (true) {
        ble.waitForEvent();
    }
}
