/*
 * Mbed Application program
 *  Nano 33 BLE Sense board runs on mbed-OS6
 *      HTS221 + LSM9DS1 + LPS22HB sensors
 *
 * Copyright (c) 2020,'21 Kenji Arai / JH1PJL
 *      http://www7b.biglobe.ne.jp/~kenjia/
 *      https://os.mbed.com/users/kenjiArai/
 *      Started:    January   22nd, 2020
 *      Revised:    February  28th, 2021
 */

//  Pre-selection --------------------------------------------------------------
#include    "select_example.h"
//#define EXAMPLE_5_COMB_ENV_SENSORS
#ifdef EXAMPLE_5_COMB_ENV_SENSORS

//  Include --------------------------------------------------------------------
#include    "mbed.h"
#include    "nano33blesense_iodef.h"
#include    "LPS22HB.h"
#include    "LSM9DS1.h"
#include    "HTS221.h"

//  Definition -----------------------------------------------------------------

//  Object ---------------------------------------------------------------------
DigitalOut      sen_pwr(PIN_VDD_ENV, 1);
DigitalOut      i2c_pullup(PIN_I2C_PULLUP, 1);
DigitalOut      led_y(PIN_YELLOW, 0);
DigitalOut      led_g(PIN_GREEN, 0);
DigitalOut      lr(PIN_LR, 1);
DigitalOut      lg(PIN_LG, 1);
DigitalOut      lb(PIN_LB, 1);
I2C             i2c(PIN_SDA1, PIN_SCL1);
LSM9DS1         *imu = NULL;
LPS22HB         *baro = NULL;
HTS221          *hum = NULL;
Timer           t;
    
//  RAM ------------------------------------------------------------------------

//  ROM / Constant data --------------------------------------------------------
const char *const msg0 = "Run Nano 33 BLE Sense module on Mbed-os";
const char *const msg1 = "Compiled on Mbed Studio:1.3.1";
const char *const msg2 = "Target is ARDUINO_NANO33BLE with Debug mode";

//  Function prototypes --------------------------------------------------------

//  subroutin (must be at this below line and do NOT change order) -------------
#include    "usb_serial_as_stdio.h"
#include    "check_revision.h"
#include    "common.h"

//------------------------------------------------------------------------------
//  Control Program
//------------------------------------------------------------------------------
int main()
{
    HTS221::HTS221_status_t aux;
    HTS221::HTS221_data_t   hum_data;

    usb_serial_initialize();
    print_revision();
    led_y = 1;
    i2c_pullup = 1;
    sen_pwr = 1;
    print_usb("%s\r\n%s\r\n%s\r\n", msg0, msg1, msg2);
    ThisThread::sleep_for(200ms);
    imu  = new LSM9DS1(PIN_SDA1, PIN_SCL1);
    baro = new LPS22HB(i2c, LPS22HB_G_CHIP_ADDR);
    hum  = new HTS221(PIN_SDA1, PIN_SCL1,HTS221::HTS221_ADDRESS, 400000);
    baro->set_odr();
    check_i2c_sensors();
    uint32_t n = 0;
    led_y = 0;
    led_g = 1;
    while(true) {
        t.reset();
        t.start();
        // Starting
        lr = 0;
        baro->get();
        hum->HTS221_SetOneShot();
        imu->begin();
        lr = 1;
        // Waiting
        lg = 0;
        do {
            thread_sleep_for(2);
            aux = hum->HTS221_GetOneShot(&hum_data);
        } while(hum_data.one_shot == HTS221::CTRL_REG2_ONE_SHOT_WAITING);
        while (baro->data_ready() == 0){
             thread_sleep_for(2);
        }
        lg = 1;
        // Reading
        lb = 0;
        hum->HTS221_GetCalibrationCoefficients (&hum_data);
        hum->HTS221_GetTemperature(&hum_data);
        hum->HTS221_GetHumidity(&hum_data);
        imu->readTemp();
        float press = baro->pressure();
        float temp = baro->temperature();
        uint32_t passed_time = chrono::duration_cast<chrono::milliseconds>(
                               t.elapsed_time()).count();
        lb = 1;
        // Showing
        lr = 0;
        lg = 0;
        lb = 0;
        print_usb("Temperature[degC]: HTS221=,%+4.1f,", hum_data.temperature);
        print_usb("LPS22HB=,%+4.1f,LSM9DS1=,%+4.1f,",
                  temp, imu->temperature_c);
        print_usb("Humidity[RH %%]: HTS221=,%3.1f,", hum_data.humidity);
        print_usb("Pressure[hPa]: LPS22H=,%6.2f,",  press);
        ++n;
        print_usb("processing time:,%2d,[ms],count:,%4d\r\n", passed_time, n);
        lr = 1;
        lg = 1;
        lb = 1;
        if (passed_time < 999){
            ThisThread::sleep_for(chrono::milliseconds(1000 - passed_time));
        }
    }
}

#endif  // EXAMPLE_5_COMB_ENV_SENSORS