/*
 * mbed library program
 *  Analog Devices / ADT7410 16-Bit Digital I2C Temperature Sensor
 *  http://www.analog.com/en/mems-sensors/digital-temperature-sensors/adt7410/products/product.html
 *
 * Copyright (c) 2014,'15,'17 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created: November   26th, 2014
 *      Revised: August     23rd, 2017
 */

#ifndef        MBED_ADT7410
#define        MBED_ADT7410

////////////// ADDRESS //////////////////////////
//  7bit address = 0b01100000(0x48) -> 8bit = 0b11000000(0x90) -> 0x91(Read) or 0x90(Write)
//  ADDR_01 = (A1=0=J4)+(A0=1=J3), ADDR_1N = (A1=1)+(A0=No Connection =0)
//      -> Please make sure your H/W configuration
#define ADT7410ADDR_00          0x90
#define ADT7410ADDR_NN          0x90
#define ADT7410ADDR_N1          0x92
#define ADT7410ADDR_01          0x92
#define ADT7410ADDR_1N          0x96
#define ADT7410ADDR_10          0x96
#define ADT7410ADDR_11          0x98

////////////// REGISTER DEFINITION //////////////
#define ADT7410_A_T_MSB         0x00
#define ADT7410_A_T_LSB         0x01
#define ADT7410_A_STATUS        0x02
#define ADT7410_A_CONFIG        0x03
#define ADT7410_A_T_H_MSB       0x04
#define ADT7410_A_T_H_LSB       0x05
#define ADT7410_A_T_L_MSB       0x06
#define ADT7410_A_T_L_LSB       0x07
#define ADT7410_A_T_C_MSB       0x08
#define ADT7410_A_T_C_LSB       0x09
#define ADT7410_A_T_HYS         0x0a
#define ADT7410_A_ID            0x0b
#define ADT7410_A_SW_RESET      0x2f

////////////// ID ///////////////////////////////
#define I_AM_ADT7410            0x19
#define GET_ID(x)               ((x >> 3) & 0x1f)
#define GET_REV(x)              (x & 0x7)

////////////// Configration /////////////////////
#define OPERATION_MODE_CONT     0x00
#define OPERATION_MODE_ONESHOT  0x20
#define OPERATION_MODE_1SPS     0x40
#define OPERATION_MODE_SHTDWN   0x60
#define RESOLUTION_13BIT        0x00
#define RESOLUTION_16BIT        0x80

/** ADT7410 class
 *
 *  Analog Devices / ADT7410 16-Bit Digital I2C Temperature Sensor
 *
 * @code
 * #include "mbed.h"
 * #include "ADT7410.h"
 *
 * // I2C Communication
 * ADT7410 t(PinName p_sda, PinName p_scl, addr);
 * // If you connected I2C line not only this device but also other devices,
 * //     you need to declare following method.
 * I2C     i2c(PinName p_sda, PinName p_scl);
 * ADT7410 t(I2C& p_i2c, addr); // default: 16bit resolution & continuous conv.
 *
 * int main() {
 *     t.set_config(OPERATION_MODE_1SPS + RESOLUTION_13BIT); // you can change the config.
 *     while(1){
 *         printf("T=%+6.3f degC\r\n", t.read_temp());
 *         wait(1.0):
 *     }
 * }
 *  @endcode
 */

class ADT7410
{
public:
    /** Configure data pin
      * @param data SDA and SCL pins
      * @param ADT7410 address (H/W configuration of A1,A0)
      */
    ADT7410(PinName p_sda, PinName p_scl, uint8_t addr);

    /** Configure data pin (with other devices on I2C line)
      * @param I2C previous definition
      * @param ADT7410 address (H/W configuration of A1,A0)
      */
    ADT7410(I2C& p_i2c, uint8_t addr);

    /** Read temperature data
      * @param none
      * @return temperature
      */
    float read_temp(void);

    /** Read status reg.
      * @param none
      * @return status register value
      */
    uint8_t read_status(void);

    /** Read configration reg.
      * @param none
      * @return configrartion register value
      */
    uint8_t read_config(void);

    /** Set configration reg.
      * @param
      * @return configrartion register value
      */
    uint8_t set_config(uint8_t cfg);

    /** Read ID
      * @param none
      * @return ID
      */
    uint8_t read_ID();

    /** Read Revision
      * @param none
      * @return Revision
      */
    uint8_t read_REV();

    /** Read one byte from specific register
      * @param register address
      * @return register data
      */
    uint8_t read_reg_byte(uint8_t reg);

    /** Write one byte into specific register
      * @param register address, data
      * @return register saved data
      */
    uint8_t write_reg_byte(uint8_t reg, uint8_t dt);

    /** check ID
      * @param none
      * @return ADT7410 = 1, others  0
      */
    uint8_t who_am_i();
    
protected:
    I2C *_i2c_p;
    I2C &_i2c;

    void read_all();
    void read_id_rev();
    void init();

private:
    uint8_t dt[4];
    char ADT7410_addr;
    uint8_t id_number;
    uint8_t rev_number;
};

#endif  //  MBED_ADT7410

