/*
 * mbed library program
 *  Analog Devices / ADT7410 16-Bit Digital I2C Temperature Sensor
 *  http://www.analog.com/en/mems-sensors/digital-temperature-sensors/adt7410/products/product.html
 *
 * Copyright (c) 2014,'15,'17 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created: November    26th, 2014
 *      Revised: FAugust     23rd, 2017
 */

#include    "mbed.h"
#include    "ADT7410.h"

ADT7410::ADT7410 (PinName p_sda, PinName p_scl, uint8_t addr)
 : _i2c_p(new I2C(p_sda, p_scl)), _i2c(*_i2c_p)
{
    ADT7410_addr = (char)addr;
    init();
}

ADT7410::ADT7410 (I2C& p_i2c, uint8_t addr)
 : _i2c(p_i2c)
{
    ADT7410_addr = (char)addr;
    init();
}

/////////////// Read All data /////////////////////////////
void ADT7410::read_all()
{
    dt[0] = ADT7410_A_T_MSB;
    _i2c.write((int)ADT7410_addr, (char *)dt, 1, true);
    _i2c.read((int)ADT7410_addr, (char *)dt, 4, false);
}

/////////////// Read Temperature //////////////////////////
float ADT7410::read_temp()
{
    uint16_t data;

    read_all();
    if (dt[3] & 0x80){  // 16bit
        data = (dt[0] << 8) + dt[1];       
    } else {            // 13bit
        data = (dt[0] << 8) + (dt[1] & 0xf8);
    }
    if (dt[0] & 0x80){  // -
        return (float)(data - 65536) / 128;
    } else {            // +
        return (float)data / 128;
    }
}

/////////////// Check Status //////////////////////////////
uint8_t ADT7410::read_status()
{
    int8_t dt = read_reg_byte(ADT7410_A_STATUS);
    return dt;
}

/////////////// Check Configration //////////////////////////
uint8_t ADT7410::read_config()
{
    int8_t dt = read_reg_byte(ADT7410_A_CONFIG);
    return dt;
}

/////////////// Set Configration //////////////////////////
uint8_t ADT7410::set_config(uint8_t cfg)
{
    return write_reg_byte(ADT7410_A_CONFIG, cfg);
}

/////////////// Read ID & REV //////////////////////////
void ADT7410::read_id_rev()
{
    uint8_t dt;

    dt = read_reg_byte(ADT7410_A_ID);
    id_number = GET_ID(dt);
    rev_number = GET_REV(dt);
}

uint8_t ADT7410::read_ID()
{
    return id_number;
}

uint8_t ADT7410::read_REV()
{
    return rev_number;
}

uint8_t ADT7410::who_am_i()
{
    if (id_number == I_AM_ADT7410){
        return 1;
    } else {
        return 0;
    }
}

/////////////// Read/Write specific register //////////////
uint8_t ADT7410::read_reg_byte(uint8_t reg)
{
    uint8_t dx[2];
    
    dx[0] = reg;
    _i2c.write((int)ADT7410_addr, (char *)dx, 1, true);
    _i2c.read((int)ADT7410_addr, (char *)dx, 1, false);
    return dx[0];
}

uint8_t ADT7410::write_reg_byte(uint8_t reg, uint8_t dt)
{
    uint8_t dx[2];
    
    dx[0] = reg;
    dx[1] = dt;
    _i2c.write((int)ADT7410_addr, (char *)dx, 2, false);
    return dx[1];
}

/////////////// Initialize ////////////////////////////////
void ADT7410::init()
{
    uint8_t dx[2];

    dx[0] = ADT7410_A_CONFIG;
    dx[1] = RESOLUTION_16BIT + OPERATION_MODE_CONT;
    _i2c.write((int)ADT7410_addr, (char *)dx, 2, false);
    read_id_rev();
}

