#include <iostream>
#include <cmath>
#include <vector>
#include <string>


#include "rgbhlr.h"
#include "mbed.h"
#include "rgbstats.h"

using namespace rgb_dws;

/*****
*	samp_mean
* Computes the means of the red, green, and blue values
* from an RGB dataset and returns it as an RGB value
*/
RGB rgb_dws::samp_mean(const std::vector<RGB>& data) {
	RGB color;

	if(data.size() == 0)
		return color;

	if(data.size() == 1) {
		color.red	= data[0].red;
		color.green = data[0].green;
		color.blue	= data[0].blue;

	} else {
	// Summate the red, green, and blue values
	for(int i = 0; i < data.size(); i++) {
		color.red		+= data[i].red; 
		color.green 	+= data[i].green;
		color.blue		+= data[i].blue;
	}
	// Compute their means
	color.red		= color.red/data.size();
	color.green		= color.green/data.size();
	color.blue		= color.blue/data.size();
	}
	return color;
}

/*****
*	median
* Computes the medians of the red, green, and blue values
* from an RGB dataset and returns it as an RGBCalc value
*/
RGBCalc rgb_dws::median(const std::vector<RGB>& data) {
	RGBCalc medians;
	int size = data.size();
	std::vector<RGB> sortedData = data;
	std::vector<int> sortedColor(data.size());

	if(size == 0) // avoid invalid address access
		return medians;

	/* Sort the red values */
	sort_dataset(sortedData, "red");
	
	/// Compute median for red values
	// If oddly-sized dataset
	if(sortedData.size() % 2 != 0)
		medians.redC = sortedData[size / 2].red;
	// If evenly-sized dataset
	else {
		medians.redC = (sortedData[(size - 1) / 2].red + sortedData[size / 2].red) / 2.0;
	}

	/* Now sort the green values */
	sort_dataset(sortedData, "green");

	/// Compute median for green values
	if(sortedData.size() % 2 != 0)
		medians.greenC = sortedData[size / 2].green;
	else {
		medians.greenC = (sortedData[(size - 1) / 2].green + sortedData[size / 2].green) / 2.0;
	}

	/* Now sort the blue values */
	sort_dataset(sortedData, "blue");

	/// Compute median for blue values
	if(sortedData.size() % 2 != 0)
		medians.blueC = sortedData[size / 2].blue;
	else {
		medians.blueC = (sortedData[(size - 1) / 2].blue + sortedData[size / 2].blue) / 2.0;
	}
	return medians;
}

/*****
*	mode	
* Computes the modes from an unsorted RGB dataset
* and returns it as an RGB value
*
* Note that multiple values with the same occurence
* will not be considered 
*/
RGB rgb_dws::mode(const std::vector<RGB>& data){
    std::vector<RGB> sortedData = data;
    RGB modes(-1,-1,-1);
  
//RED
    int n = data.size();
    
    //sorting - ASCENDING ORDER
    sort_dataset(sortedData, "red");

	int number = sortedData[0].red;
	modes.red = number;
	int count = 1;
	int countMode = 1;

	for (int i=1; i<data.size(); i++)
	{
      if (sortedData[i].red == number) 
      { // count occurrences of the current number
         ++count;
      }
      else
      { // now this is a different number
            if (count > countMode) 
            {
                  countMode = count; // mode is the biggest ocurrence
                  modes.red = number;
            }
           count = 1; // reset count for the new number
           number = sortedData[i].red;
      }
} 
/////////////////////////////////////////////////////////////////
//BLUE
    //sorting - ASCENDING ORDER
    sort_dataset(sortedData, "blue");
    
 	number = sortedData[0].blue;       
 	modes.blue = number;              
 	count = 1;
 	countMode = 1;

	for (int i=1; i<data.size(); i++)
	{
      if (sortedData[i].blue == number)             
      { // count occurrences of the current number
         ++count;
      }
      else
      { // now this is a different number
            if (count > countMode) 
            {
                  countMode = count; // biggest ocurrence
                  modes.blue = number;
            }
           count = 1; // reset count 
           number = sortedData[i].blue;
     }
	}
/////////////////////////////////////////////////////////
//GREEN
     //sorting - ASCENDING ORDER
    sort_dataset(sortedData, "green");
    
 	number = sortedData[0].green;
 	modes.green = number;
 	count = 1;
 	countMode = 1;

	for (int i=1; i<data.size(); i++)
	{
      if (sortedData[i].green == number) 
      { // count occurrences of the current number
         ++count;
      }
      else
      { // now this is a different number
            if (count > countMode) 
            {
                  countMode = count; // mode is the biggest ocurrences
                  modes.green = number;
            }
           count = 1; // reset count for the new number
           number = sortedData[i].green;
      }
	}
	return modes;
}


/*****
*	range
* Computes the ranges of the red, green, and blue values and
* returns an RGB value	
*/
RGB rgb_dws::range(const std::vector<RGB>& data) {
	RGB ranges;
	RGB max = rgb_dws::max(data);
	RGB min = rgb_dws::min(data);

	ranges.red		= max.red - min.red;
	ranges.green	= max.green - min.green;
	ranges.blue		= max.blue - min.blue;

	return ranges;
}

/*****
*	min	
* Computes the mins of the red, green, and blue values and
* returns an RGB value
*/
RGB rgb_dws::min(const std::vector<RGB>& data) {
	RGB mins;

	if(data.size() >= 1) {
		mins.red = data[0].red;
		mins.green = data[0].green;
		mins.blue = data[0].blue;

		// Retrieve minimum values for each individual color
		for(int i = 1; i < data.size(); i++) {

			if(data[i].red < mins.red)
				mins.red = data[i].red;

			if(data[i].green < mins.green)
				mins.green = data[i].green;

			if(data[i].blue < mins.blue)
				mins.blue = data[i].blue;
		}
		return mins;
	}
	return mins;
}

/*****
*	max
* Computes the maxes of the red, green, and blue values
* and returns an RGB value
*/
RGB rgb_dws::max(const std::vector<RGB>& data) {
	RGB maxes;

	if(data.size() >= 1) {
		maxes.red = data[0].red;
		maxes.green = data[0].green;
		maxes.blue = data[0].blue;

		// Retrieve minimum values for each individual color
		for(int i = 1; i < data.size(); i++) {

			if(data[i].red > maxes.red)
				maxes.red = data[i].red;

			if(data[i].green > maxes.green)
				maxes.green = data[i].green;

			if(data[i].blue > maxes.blue)
				maxes.blue = data[i].blue;
		}
		return maxes;
	}
	return maxes;
}

/*****
*	variance
* Computes the sample variance of the red, green, and blue values
* and returns an RGBCalc value
*/
RGBCalc rgb_dws::samp_variance(const std::vector<RGB>& data) {
	RGB xBar = rgb_dws::samp_mean(data);
	RGBCalc svari;

	if(data.size() >= 1)
	for(int i = 0; i < data.size(); i++) {
		svari.redC		+= (data[i].red - xBar.red) * (data[i].red - xBar.red);
		svari.greenC	+= (data[i].green - xBar.green) * (data[i].green - xBar.green);
		svari.blueC		+= (data[i].blue - xBar.blue) * (data[i].blue - xBar.blue);
	}
	svari.redC		= svari.redC / (data.size()-1);
	svari.greenC	= svari.greenC / (data.size()-1);
	svari.blueC		= svari.blueC / (data.size()-1);

	return svari;
}

/*****
*	variance
* Computes the sample standard deviation of the red, green, and blue values
* and returns and RGBCalc value
*/
RGBCalc rgb_dws::samp_stand_dev(const std::vector<RGB>& data) {

	/// Store sample variance into sStandDev
	RGBCalc sStandDev = samp_variance(data);

	/// Solve for standard deviation
	sStandDev.redC = std::sqrt(sStandDev.redC);
	sStandDev.greenC = std::sqrt(sStandDev.greenC);
	sStandDev.blueC = std::sqrt(sStandDev.blueC);

	return sStandDev;
}


/*****
*   print_distro
*       Prints frequency distributions and provides
*       some descriptive statistics of the red, green,
*       and blue values of the RGB dataset.
*
*       Class limit groupings are implemented MANUALLY
*       in order for comparisons to be made between
*       the given colors and to avoid outliers.
*/
void rgb_dws::print_distros(const std::vector<RGB>& data) {
    if(data.size() <= 1)
        return;

    RGB means   = samp_mean(data);
    RGBCalc medians = median(data);
    RGB modes 	= mode(data);
    RGB ranges  = range(data);
    RGB mins    = min(data);
    RGB maxes   = max(data);
    RGBCalc sDevs = samp_stand_dev(data);

    int lower_limit[] = {0, 37, 74, 111, 148, 185, 222};
    int upper_limit[] = {36, 73, 110, 147, 184, 221, 259};
    int k = 7; // # of classes
    std::vector<int> dis(k); // Will act as a counter for the freq distro of each color
    
    /*      Compute for Red     */
    // Compute frequency for Red
    for(int i = 0; i < k; i++) {
        dis[i] = 0;
        for(int j = 0; j < data.size(); j++) {
            if((lower_limit[i] <= data[j].red) && (data[j].red <= upper_limit[i]) ) {
                dis[i]++;
            }
        }
    }
    //Print statistics for Red
    printf("\r\n////////////////////////////////////////////////////////////////////\r\n"); 
    printf("\t\tFrequency Distribution of Red");
    printf("\r\n////////////////////////////////////////////////////////////////////\r\n");
    printf("Class Limits\t\tFrequency\t\tRelative Frequency\r\n"); 
    for(int i = 0; i < k; i++) {
        printf("%d-%d\t\t\t%d\t\t\t%.2f\r\n", lower_limit[i], upper_limit[i], 
            dis[i], (dis[i] / (double)data.size()));
    }
    printf("# of classes: %lu\r\n", k);
    printf("Sample size:  %lu\r\n\r\n", data.size());
    
    printf("Other Descriptive Statistics:\r\n");
    printf("Sample Mean:\t%d\r\n", means.red);
    printf("Minimum:\t%d\r\nMaximum:\t%d\r\nRange:\t\t%d\r\n", 
        mins.red, maxes.red, ranges.red);
    printf("Median:\t\t%.2f\r\nMode:\t\t%d\r\nSample Std Dev:\t%.4f\r\n", 
    	medians.redC, modes.red, sDevs.redC);


    /*      Compute for Green       */
    // Compute frequency for Green
    for(int i = 0; i < k; i++) {
        dis[i] = 0;
        for(int j = 0; j < data.size(); j++) {
            if((lower_limit[i] <= data[j].green) && (data[j].green <= upper_limit[i]) ) {
                dis[i]++;
            }
        }
    }
    printf("\r\n////////////////////////////////////////////////////////////////////\r\n"); 
    printf("\t\tFrequency Distribution of Green");
    printf("\r\n////////////////////////////////////////////////////////////////////\r\n");
    printf("Class Limits\t\tFrequency\t\tRelative Frequency\r\n"); 
    for(int i = 0; i < k; i++) {
        printf("%d-%d\t\t\t%d\t\t\t%.2f\r\n", lower_limit[i], upper_limit[i], 
            dis[i], (dis[i] / (double)data.size()));
    }
    printf("# of classes: %lu\r\n", k);
    printf("Sample size:  %lu  \r\n\r\n", data.size());
     
    printf("Other Descriptive Statistics:\r\n");
    printf("Sample Mean:\t%d\r\n", means.green);
    printf("Minimum:\t%d\r\nMaximum:\t%d\r\nRange:\t\t%d\r\n", 
        mins.green, maxes.green, ranges.green);
	printf("Median:\t\t%.2f\r\nMode:\t\t%d\r\nSample Std Dev:\t%.4f\r\n", 
    	medians.greenC, modes.green, sDevs.greenC);
    /*      Compute for Blue        */
    // Compute frequency for Blue
    for(int i = 0; i < k; i++) {
        dis[i] = 0;
        for(int j = 0; j < data.size(); j++) {
            if((lower_limit[i] <= data[j].blue) && (data[j].blue <= upper_limit[i]) ) {
                dis[i]++;
            }
        }
    }
    printf("\r\n////////////////////////////////////////////////////////////////////\r\n"); 
    printf("\t\tFrequency Distribution of Blue");
    printf("\r\n////////////////////////////////////////////////////////////////////\r\n");
    printf("Class Limits\t\tFrequency\t\tRelative Frequency\r\n"); 
    for(int i = 0; i < k; i++) {
        printf("%d-%d\t\t\t%d\t\t\t%.2f\r\n", lower_limit[i], upper_limit[i], 
            dis[i], (dis[i] / (double)data.size()));
    }
    printf("# of classes: %lu\r\n", k);
    printf("Sample size:  %lu\r\n\r\n", data.size());
    printf("Other Descriptive Statistics:\r\n");
    printf("Sample Mean:\t%d\r\n", means.blue);
    printf("Minimum:\t%d\r\nMaximum:\t%d\r\nRange:\t\t%d\r\n", 
        mins.blue, maxes.blue, ranges.blue);
    printf("Median:\t\t%.2f\r\nMode:\t\t%d\r\nSample Std Dev:\t%.4f\r\n", 
    	medians.blueC, modes.blue, sDevs.blueC);
    
}


/*****
*	print_reponsive_distro
*		Prints frequency distributions and provides
*		some descriptive statistics of the red, green,
*		and blue values of the RGB dataset.
*
*		Reponsive in that class limits are
*		generated in a way that considers
*		the range of values given with EACH
*		color.
*
*		Note that this function is not never called
*		in the main
*/
void rgb_dws::print_reponsive_distros(const std::vector<RGB>& data) {
	if(data.size() <= 1)
		return;

	int k = 0; // number of classes
	RGB means	= samp_mean(data);
	RGBCalc medians = median(data);
	RGB ranges	= range(data);
	RGB mins	= min(data);
	RGB maxes	= max(data);
	RGBCalc sDevs = samp_stand_dev(data);
	int classWidth = 0;
	
	/// Determine # of classes
	while(pow(2, (float)k) < data.size() ) {
		++k;
	}
	std::vector<int> lower_limit(k);
	std::vector<int> upper_limit(k);
	std::vector<int> dis(k); // Will act as a counter for the freq distro of each color
	
	/* 		Compute for Red	 	*/
	// Find upper and lower limits for red
	classWidth = ceil(ranges.red / (double)k); // Compute class width for Red
	compute_limits(lower_limit, upper_limit, classWidth, mins.red);

	// Compute frequency for red
	for(int i = 0; i < k; i++) {
		dis[i] = 0;
		for(int j = 0; j < data.size(); j++) {
			if((lower_limit[i] <= data[j].red) && (data[j].red <= upper_limit[i]) ) {
				dis[i]++;
			}
		}
	}
	//Print statistics for Red
	printf("\r\n////////////////////////////////////////////////////////////////////\r\n"); 
	printf("\t\tFrequency Distribution of Red");
	printf("\r\n////////////////////////////////////////////////////////////////////\r\n");
	printf("Class Limits\t\tFrequency\t\tRelative Frequency\r\n"); 
	for(int i = 0; i < k; i++) {
		printf("%d-%d\t\t\t%d\t\t\t%.3f\r\n", lower_limit[i], upper_limit[i], 
			dis[i], (dis[i] / (double)data.size()));
	}
	printf("# of classes: %lu\r\n\r\n", lower_limit.size());
	printf("Other Descriptive Statistics:\r\n");
	printf("Sample Mean:\t%d\r\n", means.red);
	printf("Minimum:\t%d\r\nMaximum:\t%d\r\nRange:\t\t%d\r\n", 
		mins.red, maxes.red, ranges.red);
	printf("Median:\t\t%.2f\r\nSample Std Dev:\t%.4f\r\n", medians.redC, sDevs.redC);


	/* 		Compute for Green	 	*/
	classWidth = ceil(ranges.green / (double)k);
	compute_limits(lower_limit, upper_limit, classWidth, mins.green);
	// Compute frequency for green
	for(int i = 0; i < k; i++) {
		dis[i] = 0;
		for(int j = 0; j < data.size(); j++) {
			if((lower_limit[i] <= data[j].green) && (data[j].green <= upper_limit[i]) ) {
				dis[i]++;
			}
		}
	}
	printf("\r\n////////////////////////////////////////////////////////////////////\r\n"); 
	printf("\t\tFrequency Distribution of Green");
	printf("\r\n////////////////////////////////////////////////////////////////////\r\n");
	printf("Class Limits\t\tFrequency\t\tRelative Frequency\r\n"); 
	for(int i = 0; i < k; i++) {
		printf("%d-%d\t\t\t%d\t\t\t%.3f\r\n", lower_limit[i], upper_limit[i], 
			dis[i], (dis[i] / (double)data.size()));
	}
	printf("# of classes: %lu\r\n\r\n", lower_limit.size());
	printf("Other Descriptive Statistics:\r\n");
	printf("Sample Mean:\t%d\r\n", means.green);
	printf("Minimum:\t%d\r\nMaximum:\t%d\r\nRange:\t\t%d\r\n", 
		mins.green, maxes.green, ranges.green);
	printf("Median:\t\t%.2f\r\nSample Std Dev:\t%.4f\r\n", medians.greenC, sDevs.greenC);

	/* 		Compute for Blue	 	*/
	classWidth	= ceil(ranges.blue / (double)k);
	compute_limits(lower_limit, upper_limit, classWidth, mins.blue);
	// Compute frequency for blue
	for(int i = 0; i < k; i++) {
		dis[i] = 0;
		for(int j = 0; j < data.size(); j++) {
			if((lower_limit[i] <= data[j].blue) && (data[j].blue <= upper_limit[i]) ) {
				dis[i]++;
			}
		}
	}
	printf("\r\n////////////////////////////////////////////////////////////////////\r\n"); 
	printf("\t\tFrequency Distribution of Blue");
	printf("\r\n////////////////////////////////////////////////////////////////////\r\n");
	printf("Class Limits\t\tFrequency\t\tRelative Frequency\r\n"); 
	for(int i = 0; i < k; i++) {
		printf("%d-%d\t\t\t%d\t\t\t%.3f\r\n", lower_limit[i], upper_limit[i], 
			dis[i], (dis[i] / (double)data.size()));
	}
	printf("# of classes: %lu\r\n\r\n", lower_limit.size());
	printf("Other Descriptive Statistics:\r\n");
	printf("Sample Mean:\t%d\r\n", means.blue);
	printf("Minimum:\t%d\r\nMaximum:\t%d\r\nRange:\t\t%d\r\n", 
		mins.blue, maxes.blue, ranges.blue);
	printf("Median:\t\t%.2f\r\nSample Std Dev:\t%.4f\r\n\r\n", medians.blueC, sDevs.blueC);
}


/***** (Helper Method)
*	compute_limits
*		Finds lower and upper class limits for corresponding
*		arrays within the parameter list.
*/
void rgb_dws::compute_limits(std::vector<int>& lower_limit, std::vector<int>& upper_limit, 
	int classWidth, int min) 
{
	int k = lower_limit.size();
	lower_limit[0] = min;
	/// Find the lower and upper class limits
	for(int i = 1; i < k; i++) {
		lower_limit[i] = lower_limit[i-1] + classWidth;
		upper_limit[i-1] = lower_limit[i] - 1;
	}
	upper_limit[k-1] = lower_limit[k-1] + classWidth; 
}

/***** (Helper Method)
*	sort_dataset
* Sorts a RGB datset based on the given color to be sorted
* Due to the Mbed Compiler currently only offering the C++98
* compiler, this method was required for ordering via a dataset
* via ascending
*
* Uses the selection sort algorithm 	
*/
void rgb_dws::sort_dataset(std::vector<RGB>& unsortedData, std::string color) {
	int temp, i, j;
	
	if(color == "red") {
		for(i=0; i<unsortedData.size(); i++){     
        	for(j=i+1; j<unsortedData.size(); j++){
            	if(unsortedData[i].red > unsortedData[j].red){
                	temp = unsortedData[i].red;
                	unsortedData[i].red=unsortedData[j].red;
                	unsortedData[j].red=temp;
            	}
        	}
    	}
	}
	else if(color == "green") {
		for(i=0; i<unsortedData.size(); i++){     
        	for(j=i+1; j<unsortedData.size(); j++){
            	if(unsortedData[i].green > unsortedData[j].green){
                	temp = unsortedData[i].green;
                	unsortedData[i].green=unsortedData[j].green;
                	unsortedData[j].green=temp;
            	}
        	}
    	}
	}
	else if(color == "blue") {
		for(i=0; i<unsortedData.size(); i++){     
        	for(j=i+1; j<unsortedData.size(); j++){
            	if(unsortedData[i].blue > unsortedData[j].blue){
                	temp = unsortedData[i].blue;
                	unsortedData[i].blue = unsortedData[j].blue;
                	unsortedData[j].blue = temp;
            	}
        	}
    	}
	}
}
