
#include <iostream>
#include <vector>
#include <string>

#include "mbed.h"
#include "Map.hpp"      // Mapping library
#include "TCS3200.h"    // TCS3200 color sensor library
#include "rgbhlr.h"     // RGB and RGBCalc structs
#include "rgbstats.h"   // Statistics library for RGB and RGBCalc data
using namespace rgb_dws;

/**
UART protocol for serial communication between board and PC 
*/
Serial pc(PA_2, PA_3);

/** 
Digital output pins
*/
DigitalOut led_red(PA_7);
DigitalOut led_green(PB_6);
DigitalOut led_blue(PC_7);

/**
Nucleo Board - TCS3200 Interfacing Pins

All will be considered digital output pins
besides _OUT, which is a digital input 
*/
PinName _S0     = PA_8;
PinName _S1     = PB_10;
PinName _S2     = PB_4;
PinName _S3     = PB_5;
PinName _OUT    = PA_5;

/**
Initialzing TCS3200 object
    Communication between the color sensor
    and the board is achieved with an
    user imported TCS32000 library
    (required revision due to deprectiation
    of its InterruptIn functions)
*/
 TCS3200 pw(_S0, _S1, _S2, _S3, _OUT);
    //       s0   s1   s2   s3  out

/* Delay in microseconds */
#define DELAY_0 2000000     // 2 secnods
#define DELAY_1 1000000     // 1 second
#define DELAY_2 500000      // 0.5 seconds
#define DELAY_3 2500000     // 0.25 seconds

#define HIGH    1
#define LOW     0
 

// Method signatures for methods below main
void fix_calibration_err(RGB& color);
void display_possible_color(const RGB& color);
void pw_rgb_demo(std::vector<RGB> dataSet);
void normal_inter(std::vector <RGB> dataSet);


int main() {        
    //// Initialize RGB vector to hold colors captured from the environment
    std::vector<RGB> dataSet;
   
    /// Power down color sensor initally
    pw.SetMode(TCS3200::POWERDOWN);

    int dataSetSize, selection;

    printf("\r\n\t\t|||||   ||||  |||||         ||||   |    |  ||||  \r\n");
    printf("\t\t|    | |    | |    |        |   |  |    | | \r\n");
    printf("\t\t|    | |      |||||         |    | |    |  ||||\r\n");
    printf("\t\t|||||  |  ||| |    | |||||  |    | |    |      |\r\n");
    printf("\t\t|   |  |    | |    |        |   |  | || | |    | \r\n");
    printf("\t\t|    |  ||||  |||||         ||||    |  |   ||||  \r\n");
    printf("\t             Welcome to RGB-DWS (Data With Stats)!\r\n");

    /// Terminal user input
    printf("\r\nPlease enter an action you wish to take:\r\n");
    printf("  1) [RGB Data + LED + Summarization Demonstration]\r\n");
    printf("  2) [Pulse Width Data + RGB Data + Summarization Demonstration]\r\n");
    printf("  3) [Exit the program]\r\n");
    pc.scanf("%i", &selection);

    if(selection == 1) {
        printf("\r\n  [Enter desired dataset size]:\r\n");
        pc.scanf("%i", &dataSetSize);
        
        dataSet.resize(dataSetSize);
        normal_inter(dataSet);
        return 0; 
    }
    else if(selection == 2) {
        printf("\r\n  [Enter desired dataset size]:\r\n");
        pc.scanf("%i", &dataSetSize);
        
        dataSet.resize(dataSetSize);
        pw_rgb_demo(dataSet);
    }
    else if(selection == 3) {
        printf("\r\n  [Program has been exited]\r\n");
        return 0; 
    }
    else {
        printf("\r\n  [Selection incorrect, exiting]\r\n");
        return 1;
    }
    /// Update RGB dataset size with captured input
    dataSet.resize(dataSetSize);
}


/*****
*   fix_calibratoin_err
* Fix possible RGB range errors from arbitrary calibration values
*
*/ 
void fix_calibration_err(RGB& color) {
    if(color.red < 0)
        color.red = 0;
    else if(color.red > 255)
        color.red = 255;
        
     if(color.green < 0)
        color.green = 0;
    else if(color.green > 255)
        color.green = 255;
        
     if(color.blue < 0)
        color.blue = 0;
    else if(color.blue > 255)
        color.blue = 255; 
}

/*****
*   display_possible_color
* Given a RGB value, attempts to display the closest possible LED color

    All Possible LED States:
    Red Green Blue
     0    0    0    - Occurs when values are black or close to it
     0    0    1    - Occurs when blue is larger and no colors in interval
     0    1    0    - Occurs when green is larger and no colors in interval
     0    1    1    - Occurs when green + blue fall close to one another + red smaller than both
     1    0    0    - Occurs when red is larger and no colors in interval
     1    0    1    - Occurs when red + blue fall close to one another + green smaller than both
     1    1    0    - Occurs when red + green fall close to one another + blue smaller both
     1    1    1    - Occurs when values are white or close to it
     
  Though these if conditions can be optimized, they exist in 
  their current form for readablity
  
  Note to self: when 2 states are possible, we can just
  check one if it falls under another's interval as they 
  WILL fall in each other's intervals
*/ 
void display_possible_color(const RGB& color) {
    int inter = 45;
    int mInterval = 255 - inter;
    
    // 0 0 0    Black LOW
    if(color.red <= inter && color.green <= inter 
        && color.blue <= inter) 
    {
        led_red = led_green = led_blue = LOW;
    }
    // 1 1 1    White HIGH
    else if(color.red >= mInterval && color.green >= mInterval
        && color.blue >= mInterval)
    {
        led_red = led_green = led_blue = HIGH;
    }
    // 0 0 1    Blue HIGH
    else if( (color.blue-inter > color.red) 
        && (color.blue-inter > color.green) )
    {
        led_red = led_green = LOW;
        led_blue = HIGH;
    }
    // 0 1 0    Green HIGH
    else if( (color.green-inter > color.red)
        && (color.green-inter > color.blue) )
    {
        led_red = led_blue = LOW;
        led_green = HIGH;
    }
    // 0 1 1    Green + Blue HIGH
    else if( (color.green-inter <= color.blue && color.blue <= color.green+inter)
        && (std::min(color.green, color.blue)-inter > color.red) ) // smaller interval considered
    {
        led_red = LOW;
        led_green = led_blue = HIGH;
    }
    // 1 0 0    Red HIGH
    else if( (color.red-inter > color.green)
        && (color.red-inter > color.blue) )
    {
        led_green = led_blue = LOW;
        led_red = HIGH;
    }
    // 1 0 1    Red + Blue HIGH
    else if( (color.red-inter <= color.blue && color.blue <= color.red+inter)
        && (std::min(color.red, color.blue)-inter > color.green) ) // smaller interval considered
    {
        led_green = LOW;
        led_red = led_blue = HIGH;
    }
    // 1 1 0    Red + Green HIGH
    else if( (color.red-inter <= color.green && color.green <= color.red+inter)
        && (std::min(color.red, color.green)-inter > color.blue) )
    {
        led_blue = LOW;
        led_red = led_green = HIGH;
    }
    // ELSE, they are too close to suggest any skewness
        // Treated as White HIGH
    else {
        led_red = led_blue = led_green = HIGH;
    }    
}


/*****
*   pw_rgb_demo
* Pulse width + LED + RGB + Summation Demo
*
* This function is analogous to the normal_inter function
*
* However, it not only displays the reading of the pulse widths
* of each uncalibrated RGB value 
* 
*/ 
void pw_rgb_demo(std::vector <RGB> dataSet) {
    
    long red_pw, green_pw, blue_pw;
    
    int red_min, red_max,
        green_min, green_max,
        blue_min, blue_max;
        
    int dataSize = dataSet.size();
    
    // Output frequency scaling set to 100%
    pw.SetMode(TCS3200::SCALE_100);
    
    //// Arbitrary pulse width calibration values
    // Calibrated with output frequency scaling of 100
    red_min    = 31;
    red_max    = 301;
    green_min  = 37;
    green_max  = 389;
    blue_min   = 29;
    blue_max   = 310;
    
    //// Initialize Map objects for pulse width to RGB range mapping
    Map r_map(red_min, red_max, 255, 0);
    Map g_map(green_min, green_max, 255, 0);
    Map b_map(blue_min, blue_max, 255, 0);
    
    // Provide adequate time to prepare user for capturing desired inputs
    printf("\r\n  [Reading in 3]\r\n");
    wait_us(DELAY_1);
    printf("  [Reading in 2]\r\n");
    wait_us(DELAY_1);
    printf("  [Reading in 1]\r\n");
    wait_us(DELAY_1);
    
    int i = 0;
    
    // Capture pulse widths and display them via serial port
    printf("\r\n\tPulse Widths\r\nRed\t\tGreen\t\tBlue\r\n");
    while(i < dataSize) { 
        /// Collect pulse width values
        red_pw     = (int)pw.ReadRed();
        green_pw   = (int)pw.ReadGreen();
        blue_pw    = (int)pw.ReadBlue();
        
        /// Map to RGB value range
        dataSet[i].red = (int)r_map.Calculate(red_pw);
        dataSet[i].green = (int)g_map.Calculate(green_pw);
        dataSet[i].blue = (int)b_map.Calculate(blue_pw);
        
        /// Account for calibration errors mapping pulse width range to RGB range
         fix_calibration_err(dataSet[i]);
         
        /// Output the closest possible color given an RGB data value
        display_possible_color(dataSet[i]);
        
        
        /// Display current RGB value
        printf("%d\t\t%d\t\t%d\r\n", red_pw, 
            green_pw, blue_pw);
        
        i++;
        wait_us(DELAY_2);
    }
    
    /// Power down color sensor as data has been captured
    pw.SetMode(TCS3200::POWERDOWN);
    
    char str;
    printf("\r\n[Press enter to continue]\r\n\n");
    pc.scanf("%c", &str);
    
    
    /// Print newly calibrated RGB values into serial port
    i = 0;
    while(i < dataSize) {
        /// Display current RGB value
        printf("%d\t\t%d\t\t%d\r\n", dataSet[i].red, 
            dataSet[i].green, dataSet[i].blue);
        i++;
        
    }
        
    printf("\r\n[Press enter to continue]\r\n");
    pc.scanf("%c", &str);

    print_distros(dataSet); 
}


/*****
*   normal_inter
* Normal interaction; RGB + LED + Summation Demo
*
*/ 
void normal_inter(std::vector <RGB> dataSet) {
    float red_pw, green_pw, blue_pw;
    int red_min, red_max,
        green_min, green_max,
        blue_min, blue_max;
        
    int dataSize = dataSet.size();
    
    // Output frequency scaling set to 100%
    pw.SetMode(TCS3200::SCALE_100);
    
    //// Arbitrary pulse width calibration values
    // Calibrated with output frequency scaling of 100
    red_min    = 31;
    red_max    = 301;
    green_min  = 37;
    green_max  = 389;
    blue_min   = 29;
    blue_max   = 310;
    
    //// Initialize Map objects for pulse width to RGB range mapping
    Map r_map(red_min, red_max, 255, 0);
    Map g_map(green_min, green_max, 255, 0);
    Map b_map(blue_min, blue_max, 255, 0);
    
    // Provide adequate time to prepare user for capturing desired inputs
    printf("\r\n  [Reading in 3]\r\n");
    wait_us(DELAY_1);
    printf("  [Reading in 2]\r\n");
    wait_us(DELAY_1);
    printf("  [Reading in 1]\r\n");
    wait_us(DELAY_1);
    
    int i = 0;
    printf("\r\nRed\t\tGreen\t\tBlue\r\n");
    while(i < dataSize) { 
        /// Collect pulse width values
        red_pw     = pw.ReadRed();
        green_pw   = pw.ReadGreen();
        blue_pw    = pw.ReadBlue();
        
        /// Map to RGB value range
        dataSet[i].red = (int)r_map.Calculate(red_pw);
        dataSet[i].green = (int)g_map.Calculate(green_pw);
        dataSet[i].blue = (int)b_map.Calculate(blue_pw);
        
        /// Account for calibration errors mapping pulse width range to RGB range
         fix_calibration_err(dataSet[i]);
        
        /// Output the closest possible color given an RGB data value
        display_possible_color(dataSet[i]);
        
        /// Display current RGB value
        printf("%d\t\t%d\t\t%d\r\n", dataSet[i].red, 
            dataSet[i].green, dataSet[i].blue);
        
        i++;
        wait_us(DELAY_1);
    }
    
    /// Power down color sensor as data has been captured
    pw.SetMode(TCS3200::POWERDOWN);
    
    // Set LED diodes to LOW as it is no longer in use
    led_red = led_green = led_blue = LOW;
    
    char str;
    printf("\r\n[Press enter to continue]\r\n\n");
    pc.scanf("%c", &str);
    
    print_distros(dataSet);
}

