/* MQTT Smart Sensor
 * Author: Kazuhiro Matsuda
 * Ver: 0.1 2015-01-15
 * Ver: 0.2 2015-01-18
 * Ver: 1.0 2015-01-20
 */
 
//#define DEBUG
 
#include "mbed.h"
#include "cc3000.h"
#include "TCPSocketConnection.h"
#include "PubSubClient.h"
#include "MbedJSONValue.h"
#include "main.h"
#include "ADT7410.h" 

#define STRINGIFY(x) #x
#define TO_STRING(x) STRINGIFY(x)

using namespace mbed_cc3000;

tUserFS user_info;

// CC3000 fix up
cc3000 wifi(p9, p10, p8, SPI(p5, p6, p7));

// serial console for debug
Serial pc(USBTX, USBRX);

// ADT7410 fix up
ADT7410 tempsens(p28, p27, 0x90);

#ifndef CC3000_UNENCRYPTED_SMART_CONFIG
  const uint8_t smartconfigkey[] = {0x73,0x6d,0x61,0x72,0x74,0x63,0x6f,0x6e,0x66,0x69,0x67,0x41,0x45,0x53,0x31,0x36};
#else
  const uint8_t smartconfigkey = 0;
#endif

DigitalOut led1(LED1);  // connected to AP
DigitalOut led2(LED2);  // connected to MQTT broker 
DigitalOut led3(LED3);  // publish a message
DigitalOut led4(LED4);  // subscribe a message

//MQTT broker information
char* serverIpAddr = "10.10.0.1";  
int port = 1883; 

//MQTT client information
char clientID[] = "temp1";
char pub_topic[] = "home/office/environment/temp1";
char sub_topic[] = "home/office/environment/temp1";

//Sensor information
uint16_t idx_low = 40, idx_high = 80;
#define DURATION 10

//create MQTT Client instanse
void callback(char* topic, char* payload, unsigned int len); //callback function prototype
PubSubClient mqtt(serverIpAddr, port, callback);

void callback(char* topic, char* payload, unsigned int len)
{
    MbedJSONValue json_msg, res, idx;
    std::string res_msg;
    
    // parse json object
    parse(json_msg, payload);

    if(json_msg.hasMember("cmd")) {
        led4 = 1;
        // set new threshold
        idx_low = json_msg["setidx_low"].get<int>();
        // create response
        idx["setidx_low"] = idx_low;
        res["res"] = idx;

        // serialize JSON in oder to send via network
        res_msg = res.serialize();
#ifdef DEBUG
        printf("send response %s\r\n", res_msg);
#endif
        // send MQTT message
        mqtt.publish(topic, (char*)res_msg.c_str(), strlen(res_msg.c_str()));
        led4 = 0;
    }
}

/**
 *  \brief Print cc3000 information
 *  \param none
 *  \return none
 */
void print_cc3000_info() {
    uint8_t myMAC[8];

    wifi.get_user_file_info((uint8_t *)&user_info, sizeof(user_info));
    wifi.get_mac_address(myMAC);
#ifdef DEBUG
    printf("MAC address + cc3000 info \r\n");
    printf(" MAC address %02x:%02x:%02x:%02x:%02x:%02x \r\n \r\n", myMAC[0], myMAC[1], myMAC[2], myMAC[3], myMAC[4], myMAC[5]);
    printf(" FTC        %i \r\n",user_info.FTC);
    printf(" PP_version %i.%i \r\n",user_info.PP_version[0], user_info.PP_version[1]);
    printf(" SERV_PACK  %i.%i \r\n",user_info.SERV_PACK[0], user_info.SERV_PACK[1]);
    printf(" DRV_VER    %i.%i.%i \r\n",user_info.DRV_VER[0], user_info.DRV_VER[1], user_info.DRV_VER[2]);
    printf(" FW_VER     %i.%i.%i \r\n",user_info.FW_VER[0], user_info.FW_VER[1], user_info.FW_VER[2]);
#endif
}

/**
 *  \brief Connect to SSID with a timeout
 *  \param ssid     Name of SSID
 *  \param key      Password
 *  \param sec_mode Security mode
 *  \return none
 */
void connect_to_ssid(char *ssid, char *key, unsigned char sec_mode) {
#ifdef DEBUG
    printf("Connecting to SSID: %s. Timeout is 10s. \r\n",ssid);
#endif
    if (wifi.connect_to_AP((uint8_t *)ssid, (uint8_t *)key, sec_mode) == true) {
#ifdef DEBUG
        printf(" Connected. \r\n");
#endif
    } else {
#ifdef DEBUG
        printf(" Connection timed-out (error). Please restart. \r\n");
#endif
        while(1);
  }
}

/**
 *  \brief Connect to SSID without security
 *  \param ssid Name of SSID
 *  \return none
 */
void connect_to_ssid(char *ssid) {
    wifi.connect_open((uint8_t *)ssid);
}

/**
 *  \brief First time configuration
 *  \param none
 *  \return none
 */
void do_FTC(void) {
    printf("Running First Time Configuration \r\n");
    wifi.start_smart_config(smartconfigkey);
    while (wifi.is_dhcp_configured() == false) {
         wait_ms(500);
         printf("Waiting for dhcp to be set. \r\n");
    }
    user_info.FTC = 1;
    wifi.set_user_file_info((uint8_t *)&user_info, sizeof(user_info));
    wifi._wlan.stop();
    printf("FTC finished. \r\n");
}

/**
 *  \brief Start smart config
 *  \param none
 *  \return none
 */
void start_smart_config() {
    wifi.start_smart_config(smartconfigkey);
}

void measure(){
    float tmp;
    MbedJSONValue sts, body;
    std::string msg;
    
    // measure temperature and humidity
    tmp = tempsens.read_temp();
              
    // create JSON message
    body["temp"] = tmp;
    body["unit"] = "degree C";
    sts["sts"] = body;

    // serialize JSON in oder to send via network
    msg = sts.serialize();
    led3 = 1;
    mqtt.publish(pub_topic, (char*)msg.c_str(), strlen(msg.c_str()));
    led3 = 0;
}

int main() {
    printf("----- Start MQTT client\r\n");
    pc.baud(9600);

    MbedJSONValue sts, body;
    std::string msg;
    uint16_t cnt = 0;

    // start CC3000 Wi-Fi interface
    wifi.start(0);
    
#ifdef DEBUG
    print_cc3000_info();
    printf("User's AP setup: SSID: %s, Password: %s, Security: %s \r\n", TO_STRING(SSID), TO_STRING(AP_KEY), TO_STRING(AP_SECURITY));
    printf("Attempting SSID Connection. \r\n");
#endif

    // set connection policy to manual mode
    wifi._wlan.ioctl_set_connection_policy(0, 0, 0);

    // connect to AP
    connect_to_ssid(SSID, AP_KEY, AP_SECURITY);

    while (wifi.is_dhcp_configured() == false) {
         wait_ms(500);
#ifdef DEBUG
         printf("Waiting for dhcp to be set. \r\n");
#endif
    }

#ifdef DEBUG
    tNetappIpconfigRetArgs ipinfo2;
    wifi.get_ip_config(&ipinfo2); // data is returned in the ipinfo2 structure
    printf("DHCP assigned IP Address = %d.%d.%d.%d \r\n", ipinfo2.aucIP[3], ipinfo2.aucIP[2], ipinfo2.aucIP[1], ipinfo2.aucIP[0]);
#endif
    led1 = 1;

    if(!mqtt.connect(clientID)){
        printf("\r\nConnect to server failed ..\r\n");
        return -1;
    }
    led2 = 1;
        
    // define TOPIC to read
    mqtt.subscribe(sub_topic);
        
    while(1){
        mqtt.loop();
        if(cnt > 20){
          measure();
          cnt = 0;
        }
        cnt++;
        wait(0.5);
    }
}