// ==================================================== Mar 18 2015, kayeks ==
// TextLCD_ST7032I2C.h
// ===========================================================================
// Text console library for ST7032 text LCD controller over I2C interface.

#ifndef TEXTLCD_ST7032I2C_H_
#define TEXTLCD_ST7032I2C_H_

#include "mbed.h"

#define MIN(x, y)  ((x) < (y) ? (x) : (y))

/** Text console library for ST7032 text LCD controller over I2C interface. */
class TextLCD_ST7032I2C : public Stream {
public:
    /** Bias select: 1/4 or 1/5. */
    enum Bias {
        Bias1_4    /** 1/4 bias */
        , Bias1_5  /** 1/5 bias */
    };
    
    const static uint8_t CLEAR_DISPLAY                   = 0x01;
    const static uint8_t RETURN_HOME                     = 0x02;
    const static uint8_t ENTRY_MODE_DECR_NO_SHIFT        = 0x04;
    const static uint8_t ENTRY_MODE_DECR_SHIFT           = 0x05;
    const static uint8_t ENTRY_MODE_INCR_NO_SHIFT        = 0x06;
    const static uint8_t ENTRY_MODE_INCR_SHIFT           = 0x07;
    const static uint8_t DISPLAY_OFF_CURSOR_OFF_POS_OFF  = 0x08;
    const static uint8_t DISPLAY_OFF_CURSOR_OFF_POS_ON   = 0x09;
    const static uint8_t DISPLAY_OFF_CURSOR_ON_POS_OFF   = 0x0a;
    const static uint8_t DISPLAY_OFF_CURSOR_ON_POS_ON    = 0x0b;
    const static uint8_t DISPLAY_ON_CURSOR_OFF_POS_OFF   = 0x0c;
    const static uint8_t DISPLAY_ON_CURSOR_OFF_POS_ON    = 0x0d;
    const static uint8_t DISPLAY_ON_CURSOR_ON_POS_OFF    = 0x0e;
    const static uint8_t DISPLAY_ON_CURSOR_ON_POS_ON     = 0x0f;
    const static uint8_t FUNCTION_4B_1LINE_7DOT_IS0      = 0x20;
    const static uint8_t FUNCTION_4B_1LINE_7DOT_IS1      = 0x21;
    const static uint8_t FUNCTION_4B_1LINE_11DOT_IS0     = 0x24;
    const static uint8_t FUNCTION_4B_1LINE_11DOT_IS1     = 0x25;
    const static uint8_t FUNCTION_4B_2LINE_7DOT_IS0      = 0x28;
    const static uint8_t FUNCTION_4B_2LINE_7DOT_IS1      = 0x29;
    const static uint8_t FUNCTION_4B_2LINE_11DOT_IS0     = 0x2c;
    const static uint8_t FUNCTION_4B_2LINE_11DOT_IS1     = 0x2d;
    const static uint8_t FUNCTION_8B_1LINE_7DOT_IS0      = 0x30;
    const static uint8_t FUNCTION_8B_1LINE_7DOT_IS1      = 0x31;
    const static uint8_t FUNCTION_8B_1LINE_11DOT_IS0     = 0x34;
    const static uint8_t FUNCTION_8B_1LINE_11DOT_IS1     = 0x35;
    const static uint8_t FUNCTION_8B_2LINE_7DOT_IS0      = 0x38;
    const static uint8_t FUNCTION_8B_2LINE_7DOT_IS1      = 0x39;
    const static uint8_t FUNCTION_8B_2LINE_11DOT_IS0     = 0x3c;
    const static uint8_t FUNCTION_8B_2LINE_11DOT_IS1     = 0x3d;
    const static uint8_t DDRAM                           = 0x80;
    const static uint8_t IS0_LEFT_MOVE                   = 0x10;
    const static uint8_t IS0_LEFT_SHIFT                  = 0x14;
    const static uint8_t IS0_RIGHT_MOVE                  = 0x18;
    const static uint8_t IS0_RIGHT_SHIFT                 = 0x1c;
    const static uint8_t IS0_CGRAM                       = 0x40;
    const static uint8_t IS1_BIAS5_OSC                   = 0x10;
    const static uint8_t IS1_BIAS4_OSC                   = 0x18;
    const static uint8_t IS1_ICON_ADDR                   = 0x40;
    const static uint8_t IS1_ICON_OFF_BOOST_OFF_CONTRAST = 0x50;
    const static uint8_t IS1_ICON_OFF_BOOST_ON_CONTRAST  = 0x54;
    const static uint8_t IS1_ICON_ON_BOOST_OFF_CONTRAST  = 0x58;
    const static uint8_t IS1_ICON_ON_BOOST_ON_CONTRAST   = 0x5c;
    const static uint8_t IS1_FOLLOWER_OFF_RAB            = 0x60;
    const static uint8_t IS1_FOLLOWER_ON_RAB             = 0x68;
    const static uint8_t IS1_CONTRAST                    = 0x70;
    
private:
    uint8_t**  _lineBuffer;
    uint8_t    _column, _row;
    uint8_t    _columns, _rows;
    uint8_t    _address;
    uint8_t    _osc, _rab, _contrast;
    Bias       _bias;
    I2C        _i2c;

public:
    /** Constructor of class TextLCD_ST7032I2C.
     * @param sda      I2C SPI data output (MOSI) pin.
     * @param scl      SPI clock output (SCK) pin.
     * @param address  I2C address of the LCD device.
     * @param columns  Number of characters in a row.
     * @param rows     Number of rows.
     */
    TextLCD_ST7032I2C(PinName sda, PinName scl, uint8_t address,
                      uint8_t columns, uint8_t rows);
    
    /** Destructor of class TextLCD_ST7032I2C. */
    virtual ~TextLCD_ST7032I2C();
    
    /** Hit hardware reset pin. */
    void reset();
    
    /** Initialize controller.
     * @param osc       Oscillator configuration (0..7).
     * @param rab       Rab value setting (0..7).
     * @param contrast  Contrast setting (0..63).
     * @param bias      Bias configuration for your LCD.
     */
    void init(uint8_t osc, uint8_t rab, uint8_t contrast, Bias bias);

    /** Clear display and set cursor to home. */    
    void cls();
    
    /** Set cursor position for next character.
     * @param column  Column position indexed from 0.
     * @param row     Row position indexed from 0.
     */
    void locate(uint8_t column, uint8_t row);
    
    /** Enable icon display. */
    void iconOn();
    
    /** Disable icon display. */
    void iconOff();
    
    /** Set icon display pattern.
     * @param iconAddr  Icon address (0x00..0x0f).
     * @param bits      Icon bit pattern (0x00..0x1f).
     */
    void setIcon(uint8_t iconAddr, uint8_t bits);
    
private:
    /** Implementation of putc from Stream class. */
    virtual int _putc(int c);
    
    /** Implementation of getc from Stream class. */
    virtual int _getc();

    /** Shift up the lower line for scrolling. */
    void shiftUp();

    /** Clear entire display. */
    void clear();
    
    /** Set instruction set to 0. */
    void is0();
    
    /** Set instruction set to 1. */
    void is1();
    
    /** Write a command byte for LCD controller.
     * @param c  The command byte.
     */
    void command(uint8_t c);

    /** Write a data byte for LCD controller.
     * @param d  The data byte.
     */
    void data(uint8_t d);
};

#endif
