#include <stdint.h>

#ifndef __RINGBUFFER_H_
#define __RINGBUFFER_H_

template <class T>
class RingBuffer {
public:  // For experiment use
    uint32_t readPos;
    uint32_t writePos;
    uint32_t itemCount;
    uint32_t length;
    T* buffer;
    
public:
    RingBuffer(uint32_t);
    ~RingBuffer();
    uint32_t items();
    bool isWritable();
    bool isReadable();
    bool write(T&);
    T* read();
    T* peek();
    void flush();
    int find(T);
};


template <class T>
RingBuffer<T>::RingBuffer(uint32_t len) {
    length = len;
    buffer = new T[length];
    readPos = 0;
    writePos = 0;
    itemCount = 0;
}

template <class T>
RingBuffer<T>::~RingBuffer() {
    delete[] buffer;
}

template <class T>
uint32_t RingBuffer<T>::items() {
    return itemCount;
}

template <class T>
bool RingBuffer<T>::isWritable() {
    return itemCount < length;
}

template <class T>
bool RingBuffer<T>::isReadable() {
    return itemCount > 0;
}

template <class T>
bool RingBuffer<T>::write(T& n) {
    if (!isWritable()) {
        return false;
    }
    buffer[writePos++] = n;
    if (writePos == length) {
        writePos = 0;
    }
    itemCount++;
    return true;
}

template <class T>
T* RingBuffer<T>::read() {
    uint32_t readPosTemp = readPos;
    
    if (!isReadable()) {
        return NULL;
    }
    readPos++;
    if (readPos == length) {
        readPos = 0;
    }
    itemCount--;
    return &buffer[readPosTemp];
}

template <class T>
T* RingBuffer<T>::peek() {
    if (!isReadable()) {
        return NULL;
    }
    return &buffer[readPos];
}

template <class T>
void RingBuffer<T>::flush() {
    itemCount = 0;
    readPos = writePos;
}

template <class T>
int RingBuffer<T>::find(T key) {
    uint32_t p = readPos;
    
    for (uint32_t i = 0; i < itemCount; i++) {
        if (buffer[p] == key) {
            return i;
        }
        p++;
        if (p == length) p = 0;
    }
    return -1;
}

#endif
