#include <climits>
#include "defs.h"
#include "Instrument.h"

/** Constructor of class Instrument */
Instrument::Instrument() {
    this->enableFlag = false;
    this->soundingPosition = -1;
    this->duration = 0;
    this->frequency = 0;
    this->volume = 100;
    this->expression = 127;
    this->velocity = 0;
    this->calculatePhaseDelta();
    this->calculateMasterVolume();
}

/** Destructor of class Instrument */
Instrument::~Instrument() {
}

/** Enable instrument */
void Instrument::enable() {
    this->enableFlag = true;
}

/** Disable instrument */
void Instrument::disable() {
    this->enableFlag = false;
}

/** Set sampling rate */
void Instrument::setSamplingRate(uint16_t fsamp) {
    this->samplingRate = fsamp;
}

/** Advance wave phase by one sample */
void Instrument::advancePhase() {
    this->phase += this->phaseDelta;
}

/** Reset wave phase to zero */
void Instrument::resetPhase() {
    this->phase = 0;
}

/** Set wave parameters */
void Instrument::setWave(Wavetable::wave_t w) {
    this->wave = w;
    this->phase = 0;
}

/** Set wave frequency */
void Instrument::setFrequency(uint16_t f) {
    this->frequency = f;
    calculatePhaseDelta();  // Recalculate phase delta
}

/** Set volume */
void Instrument::setVolume(uint8_t vo) {
    this->volume = vo;
    calculateMasterVolume();
}

/** Set expression */
void Instrument::setExpression(uint8_t ex) {
    this->expression = ex;
    calculateMasterVolume();
}

/** Set note velocity */
void Instrument::setVelocity(uint8_t ve) {
    this->velocity = ve;
    calculateMasterVolume();
}

/** Reset sounding duration to zero */
void Instrument::resetDuration() {
    this->duration = 0;
}

/** Get sampling rate */
uint16_t Instrument::getSamplingRate() {
    return this->samplingRate;
}

/** Get channel ID */
uint8_t Instrument::getChannelId() {
    return this->channelId;
}

/** Get enable flag */
bool Instrument::isEnable() {
    return this->enableFlag;
}

/** Get sounding note position */
int16_t Instrument::getSoundingPosition() {
    return this->soundingPosition;
}

/** Get sounding duration */
uint32_t Instrument::getDuration() {
    if (this->duration < UINT_MAX) {
        this->duration++;
    }
    return this->duration;
}

/** Get wave parameters */
Wavetable::wave_t Instrument::getWave() {
    return this->wave;
}

/** Get wave frequency */
uint16_t Instrument::getFrequency() {
    return this->frequency;
}

/** Get master volume */
uint8_t Instrument::getMasterVolume() {
    return this->masterVolume;
}

/** Get volume */
uint8_t Instrument::getVolume() {
    return this->volume;
}

/** Get expression */
uint8_t Instrument::getExpression() {
    return this->expression;
}

/** Get note velocity */
uint8_t Instrument::getVelocity() {
    return this->velocity;
}

/** Get current phase */
uint16_t Instrument::getPhase() {
    return this->phase;
}

/** Get phase delta per sample */
uint16_t Instrument::getPhaseDelta() {
    return this->phaseDelta;
}

/** (Re)calculate phase delta per sample */
void Instrument::calculatePhaseDelta() {
    // phase = 0x0000;
    phaseDelta = (uint16_t) ((uint32_t) frequency * WAVETABLE_LENGTH * 256
                             / samplingRate);
}

/** Calculate master volume from product of its volume, expression, and velocity */
void Instrument::calculateMasterVolume() {
    this->masterVolume
        = ((uint32_t) this->volume * this->expression * this->velocity) >> 16;
}
