// ColorLCDShield_Conway
// ---------------------
// A program example using Sparkfun's Color LCD Shield for FRDM-KL25Z boards.
//   - https://www.sparkfun.com/products/9363
//   - The library has modified from Sparkfun's Arduino library code
//   - Conway's Life Game simulator in a 128x128 world

#include <stdint.h>
#include "mbed.h"
#include "ColorLCDShield.h"
#include "Conway.h"

LCDShield lcd;
DigitalIn sw1(D3);
DigitalIn sw2(D4);
DigitalIn sw3(D5);
AnalogIn seed(A0);

Conway life;
Timer timer;

void drawWorld() {
    static bool firstTime = true;
    uint8_t b;
    uint16_t ix8;
    
    for (uint8_t j = 0; j < 128; j++) {
        for (uint8_t i = 0; i < 16; i++) {
            b = life.world[j][i];
            if (firstTime || b != life.worldTemp[j][i]) {
                ix8 = i << 3;
                lcd.setPixel(b & 0x80 ? 0xfff : 0x000, j, ix8);
                lcd.setPixel(b & 0x40 ? 0xfff : 0x000, j, ix8 + 1);
                lcd.setPixel(b & 0x20 ? 0xfff : 0x000, j, ix8 + 2);
                lcd.setPixel(b & 0x10 ? 0xfff : 0x000, j, ix8 + 3);
                lcd.setPixel(b & 0x08 ? 0xfff : 0x000, j, ix8 + 4);
                lcd.setPixel(b & 0x04 ? 0xfff : 0x000, j, ix8 + 5);
                lcd.setPixel(b & 0x02 ? 0xfff : 0x000, j, ix8 + 6);
                lcd.setPixel(b & 0x01 ? 0xfff : 0x000, j, ix8 + 7);
            }
        }
    }
    firstTime = false;
}

void setup() {
    sw1.mode(PullUp);
    sw2.mode(PullUp);
    sw3.mode(PullUp);
    
    lcd.init(PHILIPS);
    lcd.contrast(-63);
    lcd.clear(0x444);
    
    // Blinker ship 1
    // http://conwaylife.com/wiki/Blinker_ship_1
    uint8_t offsetX = 64 >> 3;
    uint8_t offsetY = 64;
    
    life.world[offsetY - 4][offsetX - 1] = 0x60;
    life.world[offsetY - 3][offsetX - 1] = 0xd8;
    life.world[offsetY - 2][offsetX - 1] = 0x78;
    life.world[offsetY - 1][offsetX - 1] = 0x31;
    life.world[offsetY    ][offsetX - 1] = 0x02;
    life.world[offsetY + 1][offsetX - 1] = 0x31;
    life.world[offsetY + 2][offsetX - 1] = 0x78;
    life.world[offsetY + 3][offsetX - 1] = 0xd8;
    life.world[offsetY + 4][offsetX - 1] = 0x60;
    
    life.world[offsetY - 7][offsetX    ] = 0x3c;
    life.world[offsetY - 6][offsetX    ] = 0x22;
    life.world[offsetY - 5][offsetX    ] = 0x20;
    life.world[offsetY - 4][offsetX    ] = 0x12;
    life.world[offsetY - 2][offsetX    ] = 0x80;
    life.world[offsetY - 1][offsetX    ] = 0x60;
    life.world[offsetY    ][offsetX    ] = 0x20;
    life.world[offsetY + 1][offsetX    ] = 0x60;
    life.world[offsetY + 2][offsetX    ] = 0x80;
    life.world[offsetY + 4][offsetX    ] = 0x12;
    life.world[offsetY + 5][offsetX    ] = 0x20;
    life.world[offsetY + 6][offsetX    ] = 0x22;
    life.world[offsetY + 7][offsetX    ] = 0x3c;
    
    life.world[offsetY - 1][offsetX + 1] = 0x10;
    life.world[offsetY    ][offsetX + 1] = 0x10;
    life.world[offsetY + 1][offsetX + 1] = 0x10;

    life.world[offsetY - 1][offsetX + 2] = 0xe0;
    life.world[offsetY    ][offsetX + 2] = 0xa0;
    life.world[offsetY + 1][offsetX + 2] = 0xe0;
    
    drawWorld();

    // Start timer
    timer.start();
}

void switches() {
    static bool sw1Prev, sw2Prev, sw3Prev;
    
    if (!sw1 && sw1Prev) {
        life.randomizeWorld25(seed.read_u16());
    }
    if (!sw2 && sw2Prev) {
        life.randomizeWorld(seed.read_u16());
    }
    if (!sw3 && sw3Prev) {
        life.randomizeWorld75(seed.read_u16());
    }
    
    sw1Prev = sw1;
    sw2Prev = sw2;
    sw3Prev = sw3;
}

void loop() {
    int32_t t = timer.read_ms();
    static int32_t prevSwitch = t;
    
    if (t >= prevSwitch + 20 || t < prevSwitch) {
        prevSwitch = t;
        switches();
    }
    
    life.nextGeneration();
    life.swap();
    drawWorld();
}

int main() {
    setup();

    while (1) {
        loop();
    }
}
