#include "mbed.h"

/* DA設定 */
AnalogOut da1(p18);

/* LED設定 */
DigitalOut myled1(LED1);
DigitalOut myled2(LED2);
DigitalOut myled3(LED3);
DigitalOut myled4(LED4);

/* DIO設定 */
DigitalIn inDrvAlm(p21);    // DRV-ALM
DigitalIn inReady(p22);     // READY
DigitalOut outRSTDRV(p23);  // RST-DRV
DigitalOut outPWMON(p24);   // PWM-ON

/* SIO設定 */
Serial pc(USBTX, USBRX);    // USB-SIO
Serial sio1(p9, p10);       // SIO1
Serial sio2(p13, p14);      // SIO2

/* タイマー設定 */
Timeout flipper;
Ticker led4flic;

/* 関数宣言 */
void initSIO(void);     // シリアルイニシャライズ

void cmdSIOpc(void);   // コマンド処理 pc-SIO
void cmdSIOsio2(void);  // コマンド処理 SIO1
void cmdSIOsio1(void);  // コマンド処理 SIO2

int trgADcmp(short ADdata, unsigned short ch);  // ターゲット取得AD値比較処理

unsigned short trgIO1check(unsigned short mode);    // ターゲットIO1チェック処理
unsigned short trgIO2check(unsigned short mode);    // ターゲットIO2チェック処理

short trgADreq(unsigned short ch);  // DA設定&ターゲットAD値リクエスト



/* 定数宣言 */
// DA固定出力(0-2.5-5V)
unsigned short  outDA0v0 =0;
unsigned short  outDA2v5 =19200;
//unsigned short  outDA2v5 =10000;
unsigned short  outDA5v0 =62000;

// ターゲットAD判定閾値(0-2.5-5V)
// 出力範囲:0-5V(ターゲット:-2047～2048)
short  AD0v0l =-2048;
short  AD0v0h =-1800;
short  AD2v5l =-100;
short  AD2v5h =100;
short  AD5v0l =1800;
short  AD5v0h =2047;

// SIOコマンド
// 個別
char cmdCHK[] = "CHK";  // シリアルチェック
char cmdIO1[] = "IO1";  // RST_DRV-DRV_ALMチェック
char cmdIO2[] = "IO2";  // PWM_ON-READYチェック
char cmdIOC[] = "IOC";  // DI/Oシーケンシャルチェック
char cmdDAC[] = "DAC";  // 指令電圧チェック
char cmdDAM[] = "DAM";  // 指令電圧マニュアル出力

/* 変数宣言 */
// DA用
unsigned short  u16OutDA;
short  TrgAD;

unsigned short  cmpADch;

short  SqchkNo;  // シーケンシャルチェックstep
unsigned short  SqTrgchkNo[5];  // シーケンシャルチェック時判定用ch設定
unsigned short  SqTrgDA[5];  // シーケンシャルチェック時DA値格納先(出力値)
short  SqTrgAD[5];  // シーケンシャルチェック時AD値格納先


// pc-SIO
char rx_buf_PC[50];
char rx_ch_PC;
short rx_cnt_PC;
short rx_endflg_PC;

// SIO1
char rx_buf_SIO1[50];
char rx_ch_SIO1;
short rx_cnt_SIO1;
short rx_endflg_SIO1;

// SIO2
char rx_buf_SIO2[50];
char rx_ch_SIO2;
short rx_cnt_SIO2;
short rx_endflg_SIO2;

// 汎用
short i;

int len;
int intdaout;
int intbuf;

char cbufstr[50];

// Flag
unsigned short  flgCHKrun;
unsigned short  flgRxAD;

unsigned short  flgSqADchk;


/* 割込ハンドラ */
/* pc-SIO_受信 */
void isr_Rx_PC()
{
    rx_ch_PC = pc.getc();
    rx_buf_PC[rx_cnt_PC] = rx_ch_PC;
    if( rx_ch_PC == '\r'){
        rx_endflg_PC = 1;
        myled3 = 1;
    }else{
        rx_cnt_PC++;
        myled2 = 1;
    }
}

/* SIO1_受信 */
void isr_Rx_SIO1()
{
    rx_ch_SIO1 = sio1.getc();
    rx_buf_SIO1[rx_cnt_SIO1] = rx_ch_SIO1;
    if( rx_ch_SIO1 == '\r'){
        rx_endflg_SIO1 = 1;
    }else{
        rx_cnt_SIO1++;
        myled2 = 1;
    }
}
/* SIO2_受信 */
void isr_Rx_SIO2()
{
    rx_ch_SIO2 = sio2.getc();
    rx_buf_SIO2[rx_cnt_SIO2] = rx_ch_SIO2;
    if( rx_ch_SIO2 == '\r'){
        rx_endflg_SIO2 = 1;
        myled3 = 1;
    }else{
        rx_cnt_SIO2++;
        myled2 = 1;
    }
}

/* flip_設定時間後ワンショット動作 */
void flip()
{
    led4flic.detach();
    // タイムアウトにてLED点灯
    myled4 = 1;       
    flgCHKrun = 0;
    pc.puts("TimeOut CHK\r");
    sio1.puts("NG\r");
}

/* flic_設定時間繰り返し動作 */
void flic()
{
    myled4 = !myled4;
}


/* main */
int main() {
    /* シリアル初期化 */
    initSIO();

    /* mbed上LED初期化 */
    myled2 = 0;
    myled3 = 0;
    /* 出力ポート初期化 */
    outRSTDRV = 1;  // '1'=Hi(mbed),出力=Open
    outPWMON = 1;   // '1'=Hi(mbed),出力=Open

    /* DA出力初期化 */
    da1.write_u16(0);
    flgRxAD = 0;        // ターゲットAD受信フラグ
    flgSqADchk = 0;     // シーケンシャルチェックフラグ

    SqTrgDA[0] = 0;         // 未使用
    SqTrgDA[1] = outDA0v0;  // 0.0V
    SqTrgDA[2] = outDA2v5;  // 2.5V
    SqTrgDA[3] = outDA5v0;  // 5.0V
    SqTrgDA[4] = outDA0v0;  // 0.0V

    SqTrgchkNo[0] = 0;      // 未使用
    SqTrgchkNo[1] = 1;      // 0.0V
    SqTrgchkNo[2] = 2;      // 2.5V
    SqTrgchkNo[3] = 3;      // 5.0V
    SqTrgchkNo[4] = 1;      // 0.0V

    /* 起動出力 */
    pc.printf("SIOpc test start \r");
    sio1.printf("PW ON\r");
    myled1 = 1;

    while(1){
        /* SIO処理 */
        if( rx_endflg_PC == 1 ){
            cmdSIOpc();   // コマンド処理 pc-SIO
        }else if( rx_endflg_SIO1 == 1 ){
            cmdSIOsio1();  // コマンド処理 SIO1
        }else if( rx_endflg_SIO2 == 1 ){
            cmdSIOsio2();  // コマンド処理 SIO2
        }
        
        /* AD値判定処理 */
        if( flgRxAD == 1){
            if( flgSqADchk == 0 ){
                /* AD値 個別判定 */
                if( trgADcmp(TrgAD, cmpADch) == 0){
                    // OK
                    sprintf(cbufstr,"OK,%05d\r",(int)TrgAD);
                }else{
                    // NG
                    sprintf(cbufstr,"NG,%05d\r",(int)TrgAD);
                }
                sio1.puts(cbufstr);
            }else{
                /* AD値 シーケンシャル判定 */
                SqTrgAD[SqchkNo] = TrgAD;   // 受信AD値保存
                if( trgADcmp(SqTrgAD[SqchkNo], SqTrgchkNo[SqchkNo]) == 0){
                    // OK
                    switch(SqchkNo){
                        case 1:
                            // 指令値2.5V設定&ターゲットモニタ値リクエスト
                            trgADreq(2);
                            SqchkNo++;
                            break;
                        case 2:
                            // 指令値5.0V設定&ターゲットモニタ値リクエスト
                            trgADreq(3);
                            SqchkNo++;
                            break;
                        case 3:
                            // 指令値0.0V設定&ターゲットモニタ値リクエスト
                            trgADreq(1);
                            SqchkNo++;
                            break;
                        case 4:
                            // シーケンシャル処理終了
                            sprintf(cbufstr,"OK\r");
                            sio1.puts(cbufstr);
                            for(i=1;i<=4;i++){
                                sprintf(cbufstr,"%05d,%05d\r",(int)SqTrgDA[i],(int)SqTrgAD[i]);
                                sio1.puts(cbufstr);
                            }
                            flgSqADchk = 0; //シーケンシャルチェック終了
                            SqchkNo = 0; //シーケンシャルチェックstepリセット
                        default:
                        // シーケンシャル処理終了
                        flgSqADchk = 0; //シーケンシャルチェック終了
                        SqchkNo = 0; //シーケンシャルチェックstepリセット
                        break;
                    }
                }else{
                    // NG
                    sprintf(cbufstr,"NG,%05d\r",(int)SqchkNo);
                    sio1.puts(cbufstr);
                    for(i=1;i<=SqchkNo;i++){
                        sprintf(cbufstr,"%05d,%05d\r",(int)SqTrgDA[i],(int)SqTrgAD[i]);
                        sio1.puts(cbufstr);
                    }
                    flgSqADchk = 0; //シーケンシャルチェック終了
                    SqchkNo = 0; //シーケンシャルチェックstepリセット
                }
            }
        }
    }
}


/* 関数 */
/* シリアルイニシャライズ */
void initSIO(void)
{
    /* シリアル-割込ハンドラ登録 */
    /* pc-SIO */
    pc.attach(isr_Rx_PC,Serial::RxIrq); // 受信割込ハンドラ
    /* SIO1 */
    sio1.attach(isr_Rx_SIO1,Serial::RxIrq); // 受信割込ハンドラ
    /* SIO2 */
    sio2.attach(isr_Rx_SIO2,Serial::RxIrq); // 受信割込ハンドラ
 
    /* シリアル初期化 */
    /* pc-SIO */
    pc.baud(38400);
    pc.format(8, SerialBase::None ,1);
    /* pc-SIO */
    sio1.baud(38400);
    sio1.format(8, SerialBase::None ,1);
    /* pc-SIO */
    sio2.baud(38400);
    sio2.format(8, SerialBase::None ,1);

    /* pc-SIO チェック用初期化 */
    rx_cnt_PC = 0;
    rx_endflg_PC = 0;
    /* SIO1 チェック用初期化 */
    rx_cnt_SIO1 = 0;
    rx_endflg_SIO1 = 0;
    /* SIO2 チェック用初期化 */
    rx_cnt_SIO2 = 0;
    rx_endflg_SIO2 = 0;

    /* 受信領域初期化 */
    memset(cbufstr,'\0',sizeof(cbufstr));
    memset(rx_buf_PC,'\0',sizeof(rx_buf_PC));
    memset(rx_buf_SIO1,'\0',sizeof(rx_buf_SIO1));
    memset(rx_buf_SIO2,'\0',sizeof(rx_buf_SIO2));
}


/* コマンド処理 */
/* 20141008 現状ではターミネート受信後返信処理 */
/* SIO-pc */
void cmdSIOpc(void)
{
    memset(cbufstr,'\0',sizeof(cbufstr));

    /* 文字長チェック */
    len = strlen(rx_buf_PC);
    strncpy(cbufstr,rx_buf_PC,3);
    
    /* コマンド判定 */
    if( strlen(rx_buf_PC) <= 4){
        if(strcmp(cbufstr,cmdCHK) == 0){
            //  コマンド可
            pc.puts("Res CHK\r");
            flipper.detach();
            led4flic.detach();
            if( flgCHKrun == 1){
                flgCHKrun = 0;
                myled4 = 0;
                sio1.puts("OK\r");
            }
        }else{
            //  コマンド不可
            pc.puts("No CHK\r");
        }
    }else if( strlen(rx_buf_PC) <= 10){
        if(strcmp(cbufstr,cmdDAC) == 0){
            // コマンド可 DAC
            strncpy(cbufstr,&rx_buf_PC[4],5);
            intbuf = atoi( cbufstr );
            TrgAD = (short)intbuf;
            flgRxAD = 1;
        }else{
            //  コマンド不可
            pc.puts("No DAC\r");
        }

    }else{
        pc.puts("Lengs error\r");
        pc.puts(cbufstr);
    }

    memset(rx_buf_PC,'\0',sizeof(rx_buf_PC));
    rx_cnt_PC = 0;
    myled2 = 0;
    myled3 = 0;
    rx_endflg_PC = 0;
}

/* SIO1 */
void cmdSIOsio1(void)
{
    memset(cbufstr,'\0',sizeof(cbufstr));
    /* 文字長チェック */
    len = strlen(rx_buf_SIO1);
    /* コマンド判定 */
    strncpy(cbufstr,rx_buf_SIO1,3);
    if( len == 4){
        if(strcmp(cbufstr,cmdCHK) == 0){
            // コマンド可:CHK
            if( flgCHKrun == 1){
                flipper.detach();
                led4flic.detach();
            }
            flipper.attach(&flip,5.0);
            led4flic.attach(&flic,0.3);
            flgCHKrun = 1;
            sio2.puts("CHK\r");
        }else if(strcmp(cbufstr,cmdIOC) == 0){
            // コマンド可:IOC
            if( trgIO1check(0) == 0){                       // IO1 = OFF check
                if( trgIO1check(1) == 0){                   // IO1 = ON check
                    if( trgIO1check(0) == 0){               // IO1 = OFF check
                        if( trgIO2check(0) == 0){           // IO2 = OFF check
                            if( trgIO2check(1) == 0){       // IO2 = ON check
                                if( trgIO2check(0) == 0){   // IO2 = OFF check
                                    sio1.puts("OK\r");
                                }else sio1.puts("NG.6\r");
                            }else sio1.puts("NG.5\r");
                        }else sio1.puts("NG.4\r");
                    }else sio1.puts("NG.3\r");
                }else sio1.puts("NG.2\r");
            }else sio1.puts("NG.1\r");
        }else{
            // コマンド不可
            sio1.puts("CMD Error 3\r");
        }
    }else if( len == 6){
        if(strcmp(cbufstr,cmdIO1) == 0){
            // コマンド可:IO1
//            sio1.puts("res IO1\r");
            if( rx_buf_SIO1[4] == '0' ){
                if( trgIO1check(0) == 0){
                    sio1.puts("OK\r");
                }else sio1.puts("NG\r");
            }else if(rx_buf_SIO1[4] == '1'){
                if( trgIO1check(1) == 0){
                    sio1.puts("OK\r");
                }else sio1.puts("NG\r");
            }else{
                sio1.puts("Data Error cmdIO1\r");
            }
        }else if(strcmp(cbufstr,cmdIO2) == 0){
            // コマンド可:IO2
//            sio1.puts("res IO2\r");
            if( rx_buf_SIO1[4] == '0' ){
                if( trgIO2check(0) == 0){
                    sio1.puts("OK\r");
                }else sio1.puts("NG\r");
            }else if(rx_buf_SIO1[4] == '1'){
                if( trgIO2check(1) == 0){
                    sio1.puts("OK\r");
                }else sio1.puts("NG\r");
            }else{
                sio1.puts("Data Error cmdIO2\r");
            }
        }else if(strcmp(cbufstr,cmdDAC) == 0){
            // コマンド可:DAC
//            sio1.puts("res DAC\r");
            if(rx_buf_SIO1[4] == '1'){
                // 指令値0.0V設定&ターゲットモニタ値リクエスト
                trgADreq(1);
            }else if(rx_buf_SIO1[4] == '2'){
                // 指令値2.5V設定&ターゲットモニタ値リクエスト
                trgADreq(2);
            }else if(rx_buf_SIO1[4] == '3'){
                // 指令値5.0V設定&ターゲットモニタ値リクエスト
                trgADreq(3);
            }else if(rx_buf_SIO1[4] == '4'){
                // 格納先リセット
                for(i=0;i<=4;i++){
                    SqTrgAD[i] = 0;
                }
                // シーケンシャルチェックフラグ = ON
                flgSqADchk = 1;
                // シーケンシャルチェックstep = 1
                SqchkNo = 1;
                // 指令値0.0V設定&ターゲットモニタ値リクエスト
                trgADreq(1);
            }else{
                sio1.puts("Data Error cmdDAC\r");
            }
        }else{
            // コマンド不可
            sio1.puts("CMD Error 5\r");
            sio1.printf("SIO1Rx= %s ",rx_buf_SIO1);
        }
    }else if( len == 10){
        if(strcmp(cbufstr,cmdDAM) == 0){
            // コマンド可:DAM
//            sio1.puts("res DAM\r");
            sio1.puts("DAM\r");
            strncpy(cbufstr,&rx_buf_SIO1[4],5);
            intdaout = atoi( cbufstr );
            // DA出力設定
            da1.write_u16((unsigned short)intdaout);
            
        }else{
            // コマンド不可
            sio1.puts("CMD Error 9\r");
        }
    }else{
        sio1.puts("CMD Lengs Error\r");
    }

    memset(rx_buf_SIO1,'\0',sizeof(rx_buf_SIO1));
    rx_cnt_SIO1 = 0;
    myled2 = 0;
    myled3 = 0;
    rx_endflg_SIO1 = 0;
}

/* SIO2 */
void cmdSIOsio2(void)
{
    memset(cbufstr,'\0',sizeof(cbufstr));

    /* 文字長チェック */
    len = strlen(rx_buf_SIO2);
    strncpy(cbufstr,rx_buf_SIO2,3);
    
    /* コマンド判定 */
    if( strlen(rx_buf_SIO2) <= 4){
        if(strcmp(cbufstr,cmdCHK) == 0){
            //  コマンド可
            pc.puts("Res CHK\r");
            flipper.detach();
            led4flic.detach();
            if( flgCHKrun == 1){
                flgCHKrun = 0;
                myled4 = 0;
                sio1.puts("OK\r");
            }
        }else{
            //  コマンド不可
            pc.puts("No CHK\r");
        }
    }else if( strlen(rx_buf_SIO2) <= 10){
        if(strcmp(cbufstr,cmdDAC) == 0){
            // コマンド可 DAC
            strncpy(cbufstr,&rx_buf_SIO2[4],5);
            intbuf = atoi( cbufstr );
            TrgAD = (short)intbuf;
            flgRxAD = 1;
        }else{
            //  コマンド不可
            pc.puts("No DAC\r");
        }

    }else{
        pc.puts("Lengs error\r");
        pc.puts(cbufstr);
    }

    memset(rx_buf_SIO2,'\0',sizeof(rx_buf_SIO2));
    rx_cnt_SIO2 = 0;
    myled2 = 0;
    myled3 = 0;
    rx_endflg_SIO2 = 0;
}


/* ターゲットIO1チェック処理                                    */
/* 引数:                                                     */
/*      unsigned short mode     :処理条件(0=OFF,1=ON)         */
/* 戻値:                                                     */
/*      unsigned short 0:OK,1:NG,2:mode error               */
/*  */
unsigned short trgIO1check(unsigned short mode)
{
    if( mode == 0 ){
        outRSTDRV = 1;
        wait(0.5);
        if( inDrvAlm == 1 ){
            return 0;
        }else{
            return 1;
        }
    }else if( mode == 1){
        outRSTDRV = 0;
        wait(0.5);
        if( inDrvAlm == 0 ){
            return 0;
        }else{
            return 1;
        }
    }else{
        return 2;
    }
}


/* ターゲットIO2チェック処理                                    */
/* 引数:                                                     */
/*      unsigned short mode     :処理条件(0=OFF,1=ON)         */
/* 戻値:                                                     */
/*      unsigned short 0:OK,1:NG,2:mode error    */
/*  */
unsigned short trgIO2check(unsigned short mode)
{
    if( mode == 0 ){
        outPWMON = 1;
        wait(0.5);
        if( inReady == 1 ){
            return 0;
        }else{
            return 1;
        }
    }else if( mode == 1){
        outPWMON = 0;
        wait(0.5);
        if( inReady == 0 ){
            return 0;
        }else{
            return 1;
        }
    }else{
        return 2;
    }
}


/* ターゲットAD値判定処理 */
/* 変数名:int trgADcmp(short ADdata, unsigned short ch) */
/* 引数:                                                          */
/*      unsigned short ADdata   :判定値(受信ADデータ)               */
/*      unsigned short ch       :判定レンジ指定[1:0V0,2:2v5,3,5v0]  */
/* 戻値:int 0:OK, 1:NG                                            */
/*  */
int trgADcmp(short ADdata, unsigned short ch)
{
    // フラグリセット
    flgRxAD = 0;

    memset(cbufstr,'\0',sizeof(cbufstr));

    // 比較範囲チェック
    if( ch == 1 ){
        // 0V比較
        if( ADdata >= AD0v0l && ADdata <= AD0v0h){
            // OK
            return 0;
        }else{
            // NG
            return 1;
        }
    }else if( ch == 2 ){
        // 2V5比較
        if( ADdata >= AD2v5l && ADdata <= AD2v5h){
            // OK
            return 0;
        }else{
            // NG
            return 1;
        }
    }else if( ch == 3 ){
        // 5V0比較
        if( ADdata >= AD5v0l && ADdata <= AD5v0h){
            // OK
            return 0;
        }else{
            // NG
            return 1;
        }
    }else{
        // ch error
        sio1.puts("error ADch\r");
        return 1;
    }
}


/* ターゲットAD値出力設定&モニタ値リクエスト処理 */
/* 変数名:short trgADreq(unsigned short ch) */
/* 引数:                                                          */
/*      unsigned short ch       :判定レンジ指定[1:0V0,2:2v5,3,5v0]  */
/* 戻値:short 0:OK, 1:NG                                            */
/*  */
short trgADreq(unsigned short ch)
{
    // DA 0.0V設定
    if( ch == 1 ){
//        sprintf(cbufstr,"DAC,%05d\r",outDA0v0);
//        sio1.puts(cbufstr);
        da1.write_u16(outDA0v0);
    // DA 2.5V設定
    }else if( ch == 2 ){
//        sprintf(cbufstr,"DAC,%05d\r",outDA2v5);
//        sio1.puts(cbufstr);
        da1.write_u16(outDA2v5);
    // DA 5.0V設定
    }else if( ch == 3 ){
//        sprintf(cbufstr,"DAC,%05d\r",outDA5v0);
//        sio1.puts(cbufstr);
        da1.write_u16(outDA5v0);
    }
    
    // wait(0.2sec)後、trgAD値リクエスト
    wait(0.2);
    sio2.puts("DAC\r");

    // DA 0.0V設定
    if( ch == 1 ){
        cmpADch = 1;    // 比較用chセット
        return 0;
    // DA 2.5V設定
    }else if( ch == 2 ){
        cmpADch = 2;    // 比較用chセット
        return 0;
    // DA 5.0V設定
    }else if( ch == 3 ){
        cmpADch = 3;    // 比較用chセット
        return 0;
    }else{
        return 1;
    }
}