//I2C.h
#ifndef CPP_TO_PIGPIO_I2C_HEADER
#define CPP_TO_PIGPIO_I2C_HEADER
namespace CPPToPigpio{
	/** The I2C interface provides I2C Master functionality.
	 */
    class I2C: public CPPToPigpio{
        private:
            short _pinsda;
			short _pinscl;
            int _clk;
			int _handle;//current i2c device handle
			unsigned _I2CBus; //1 or 0 depending on bus being used (cannot mix)
			bool _repeated;
			int _baud;
			bool _bbstarted;
			
		public:
			/** Create an I2C Master interface, connected to the specified pins
			 *
			 *  @param sda I2C data line pin
			 *  @param scl I2C clock line pin
			 */
			I2C(PinName sda, PinName scl); //Construct I2C
			
			/** Create an I2C Master interface, connected to the specified pins
			 *
			 *  @param sda I2C data line pin
			 *  @param scl I2C clock line pin
			 */
			I2C(int sda, int scl); //Used for int args
			
			/** Set the frequency of the I2C clock frequency for bit banging
			 *
			 *  @param hz The bus frequency in hertz
			 */
			void frequency(int hz);
			
			/** Read from an I2C slave
			 *
			 * Performs a complete read transaction. 
			 * Read "length" bytes from device at "address" into "buf". The bottom bit of
			 * the address is forced to 1 to indicate a read.
			 *
			 *  @param address 8-bit I2C slave address [ addr | 1 ]
			 *  @param data Pointer to the byte-array to read data in to
			 *  @param length Number of bytes to read
			 *  @param repeated Repeated start, true - don't send stop at end
			 *
			 *  @returns
			 *       0 on success (ack),
			 *   non-0 on failure (nack)
			 */
			int read(unsigned address, char* buf, int length, bool repeated); //multi-byte read

			/** Read a single byte from the I2C bus
			 *
			 *  @param ack indicates if the byte is to be acknowledged (1 = acknowledge)
			 *
			 *  @returns
			 *    the byte read
			 */
			int read(int ack);//single-byte read (??)

			/** Write to an I2C slave
			 *
			 * Performs a complete write transaction. 
			 * Write "length" bytes from buf to device at "address". The bottom bit of
			 * the address is forced to 0 to indicate a write.
			 *
			 *  @param address 8-bit I2C slave address [ addr | 0 ]
			 *  @param data Pointer to the byte-array data to send
			 *  @param length Number of bytes to send
			 *  @param repeated Repeated start, true - do not send stop at end
			 *
			 *  @returns
			 *       0 on success (ack),
			 *   non-0 on failure (nack)
			 */
			int write(unsigned address, char* buf, int length, bool repeated); //Set the value of the output
			
			/** Write single byte out on the I2C bus
			 *
			 *  @param data data to write out on bus
			 *
			 *  @returns
			 *    '0' - NAK was received
			 *    '1' - ACK was received,
			 *    '2' - timeout
			 */
			int write(int data); //single-byte write
			
			/** Creates a start condition on the I2C bus
			*/	
			void start(void);

			/** Creates a stop condition on the I2C bus
     		*/
			void stop(void);

			/** Allows construction of arbitrary I2C command. 
			 * Basically a wrapper for pigpio's bbi2cZip
			 * 
			 * @param data Pointer to the byte-array to read data in to
			 * @param length Number of bytes to read
			 * @param data Pointer to the byte-array data to send
			 * @param length Number of bytes to send
			 * @param repeated Repeated start, true - do not send stop at end
			 */
			int bang(char* inBuf, unsigned inLen, char* outBuf, unsigned outLen, bool cont);
			int writeAddr(unsigned address);
			~I2C();

			enum cmdEnum{End=0x00, Escape=0x01, Start=0x02, Stop=0x03, Addr=0x04, Flags=0x05, Read=0x06, Write=0x07};
			
    };
}

#endif
