/*
    Radar.h - Simple radar simulator, example of graphic commands usage to create 2D graphics.

    Copyright(c) 2016 karpent at gmail.com, MIT License

    Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files(the "Software"),
    to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and / or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions :

    The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
    OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
    THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#pragma once

#include "Commons.h"
#include "Display.h"
#include "Target.h"
#include <list>

using namespace std;

//static const uint8_t _maxRangeNumber = 1;
//static const uint8_t _minRangeNumber = 0;

#define MAX_RANGE_INDEX 2
static const float _ranges[MAX_RANGE_INDEX + 1]             = { 50, 100, 300 }; // km
static const float _rangeMarkersDelta[MAX_RANGE_INDEX + 1]  = { 10,  25,  50 }; // km
static const int azimuthMarkersCount = 8;

/**
  * @brief Simple radar simulator
  */
class Radar
{
public:

    Radar(Display* display);
    ~Radar();

    /// <summary>
    /// Draws all visible tracks.
    /// </summary>
    void DrawTracks();


    /// <summary>
    /// Updates the targets in sector for the current time.
    /// Returns true if targets found.
    /// </summary>
    /// <param name="startAngle">The start angle.</param>
    /// <param name="endAngle">The end angle.</param>
    /// <param name="currentTime">Current Time in miliseconds.</param>
    bool UpdateTargetsLocation(float startAngle, float endAngle, uint32_t currentTime);


    /// <summary>
    /// Gets currently selected range in world coordinates [km].
    /// </summary>
    /// <returns></returns>
    static float GetRange() {
        return _ranges[_rangeNumber];
    }

    /// <summary>
    /// Gets the maximum radar range in world coordinates [km].
    /// </summary>
    /// <returns></returns>
    static float GetMaxRange() {
        return _ranges[MAX_RANGE_INDEX];
    }

    /// <summary>
    /// Sets current range number.
    /// </summary>
    /// <param name="rangeNumber">The range number.</param>
    void SetRange(uint8_t rangeNumber);

    /// <summary>
    /// Increase or decrease range.
    /// </summary>
    /// <param name="change">1 for increase range, -1 for decrease range</param>
    bool ChangeRange(int change);    

    static int GetCenterX() {
        return _centerX;
    }

    static int GetCenterY() {
        return _centerY;
    }

    static void SetCenter(int x, int y) {
        _centerX = x;
        _centerY = y;
    }

protected:
    /// <summary>
    /// Initializes this instance.
    /// </summary>
    void Initialize();

    Display* GetDisplay();
    list<Target> targetsList;
    
    void DrawPlot(Location* plot);
    void DrawRadarBeam(float azimuth);
    void DrawBorder();
    void DrawMarkers();
    void DrawTarget(Target* target);
    void DrawRangeButtons(uint16_t raSize);

    /// <summary>
    /// Display window in local coordinates system
    /// </summary>
    Window window;

private:

    /// <summary>
    /// Center y position in local coordinate system
    /// </summary>
    static int _centerX;

    /// <summary>
    /// Center y position in local coordinate system
    /// </summary>
    static int _centerY;

    /// <summary>
    /// Selected range (0..MaxRangeNumber)
    /// </summary>
    static uint8_t _rangeNumber;

    static float _scaleRatio;

    // Can be canvas, display or SDL renderer
    Display* _display;
    
};