//
// RK043FN48H.h - Graphics commands for DISCO_F746NG display build on top of stm32746g driver.
//

#pragma once

#include "Commons.h"
#include "Display.h"
#include "stm32746g_discovery_lcd.h"
#include "GrFont.h"

#define ALPHA_MASK 0x00FFFFFF

/**
 * @brief Display layers available for RK043FN48H
 */
typedef enum ELayer {
    Background = 0, // Bottom
    Foreground = 1  // Upper
} Layer;

/**
  * @brief DISCO_F746NG display
  */
class RK043FN48H : public Display
{
public:
    RK043FN48H();
    ~RK043FN48H();

    /**
    * @brief  Clear the active layer using background color.
    * @retval None
    */
    void virtual Clear();

    /**
    * @brief  Clear the active layer using given color.
    * @retval None
    */
    void Clear(uint32_t color);

    /**
    * @brief Clear selected layer using given color.
    * @retval None
    */
    void ClearLayer(Layer layer, uint32_t color);

    /**
    * @brief  Set background color for active layer.
    * @retval None
    */
    void SetBackgroundColor(uint32_t color);

    /**
    * @brief  Set foreground color for active layer.
    * @retval None
    */
    void SetForegroundColor(uint32_t color);

    /// <summary>
    /// Sets the color of the draw.
    /// </summary>
    /// <param name="red">The red.</param>
    /// <param name="green">The green.</param>
    /// <param name="blue">The blue.</param>
    /// <param name="alpha">The alpha.</param>
    void virtual SetDrawColor(uint8_t red, uint8_t green, uint8_t blue, uint8_t alpha);

    /// <summary>
    /// Gets the color of the draw.
    /// </summary>
    /// <returns>Draw color value</returns>
    uint32_t virtual GetDrawColor();

    /// <summary>
    /// Sets the clear color.
    /// </summary>
    /// <param name="red">The red.</param>
    /// <param name="green">The green.</param>
    /// <param name="blue">The blue.</param>
    /// <param name="alpha">The alpha.</param>
    void virtual SetClearColor(uint8_t red, uint8_t green, uint8_t blue, uint8_t alpha);

    /// <summary>
    /// Gets the clear color.
    /// </summary>
    /// <returns>Clear color value</returns>
    uint32_t virtual GetClearColor();

    /// <summary>
    /// Draws the point.
    /// </summary>
    /// <param name="posX">The position x.</param>
    /// <param name="posY">The position y.</param>
    /// <param name="colorMask">The color mask.</param>
    void virtual DrawPoint(int posX, int posY, uint32_t colorMask);

    //void virtual DrawLine(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2);

    //void virtual FillRectangle(uint16_t posX, uint16_t posY, uint16_t x2, uint16_t y2);

    void CopyBitmap(Layer layer, uint8_t *bitmap, uint32_t width, uint32_t height, uint32_t rgbGolorCode);

    /// <summary>
    /// Returns screen width.
    /// </summary>
    /// <returns></returns>
    uint16_t virtual DisplayWidth();

    /// <summary>
    /// Returns screen height.
    /// </summary>
    /// <returns></returns>
    uint16_t virtual DisplayHeight();

    void SetActiveLayer(Layer layer);

    Layer GetActiveLayer();

    void SetLayersTransparency( uint8_t background, uint8_t foreground);

    void SetLayersVisibility( bool background, bool foreground);

//==============================================================================
//
//  Text output
//
//==============================================================================

    /**
    * @brief  Change current text cursor location for the given position on the screen.
    *         Remark - cursor is unvisible.
    * @param  x : X position
    * @param  y : Y position
    * @retval None
    */
    void GotoXY(int x, int y);

    /**
    * @brief  Put character on the screen using current font and cursor location.
    *         See also: gotoXY(x,y), SetFont(type), SetForegroundColor(c), SetBackgroundColor(c)
    * @param  ch: Character to print
    * @retval None
    */
    void putch(char ch);

    /**
    * @brief  Put string on the screen using current font and cursor location.
    *         See also: gotoXY(x,y), SetFont(type), SetForegroundColor(c), SetBackgroundColor(c)
    * @param  str: C string (char *) to print.
    * @retval None
    */
    void puts(const char * str);

    /**
    * @brief  Draw string on the screen at given position using current font.
    *         See also: SetFont(type), SetForegroundColor(c), SetBackgroundColor(c)
    * @param  str: C string (char *) to print.
    * @retval None
    */
    void virtual DrawText(int posX, int posY, char * str);

private:

    // Frame buffer addresses for layers
    uint32_t FbForegroundStartAdress;
    uint32_t FbBackgroundStartAdress;

    /**
    *   @brief Mirror structure for drivers DropProp,
    *   Contains selected font, background and foreground color
    */
    LCD_DrawPropTypeDef actualDrawProp[MAX_LAYER_NUMBER];

    /**
    *   @brief Actual cursor text position for text output
    */
    Point _cursorPos[MAX_LAYER_NUMBER];

    /**
    *   @brief Current fonts for layers
    */
    GrFont _selectedFont[MAX_LAYER_NUMBER];

    /**
    *   @brief Internal method to move text coursor to the next line
    *       depending on font height. Font width is not used.
    *   @param w : Font width
    *   @param h : Font height
    */
    void gotoNewLine(int w, int h);

};

