//
// RK043FN48H.h - DISCO_F746NG display
//

#include "RK043FN48H.h"

static Layer _selectedLayer = Foreground;

RK043FN48H::RK043FN48H()
{
    BSP_LCD_Init();

    FbBackgroundStartAdress = LCD_FB_START_ADDRESS;
    FbForegroundStartAdress = LCD_FB_START_ADDRESS+(BSP_LCD_GetXSize()*BSP_LCD_GetYSize()*4);

    BSP_LCD_LayerDefaultInit(0, FbBackgroundStartAdress);
    BSP_LCD_LayerDefaultInit(1, FbForegroundStartAdress);

    BSP_LCD_DisplayOn();

    actualDrawProp[0].TextColor = LCD_COLOR_WHITE;
    actualDrawProp[0].BackColor = LCD_COLOR_BLACK;
    actualDrawProp[0].pFont = &Font16; // &LCD_DEFAULT_FONT
    _selectedFont[0] = GrFont(Courier16);

    actualDrawProp[1].TextColor = LCD_COLOR_WHITE; // & ALPHA_MASK;
    actualDrawProp[1].BackColor = LCD_COLOR_BLACK & ALPHA_MASK;
    actualDrawProp[1].pFont = &Font16; // &LCD_DEFAULT_FONT
    _selectedFont[1] = GrFont(Courier16);

    // Initialize layer 0 properties
    SetActiveLayer(Background);
    SetBackgroundColor(actualDrawProp[0].BackColor);
    Clear();
    SetForegroundColor(actualDrawProp[0].TextColor);
    BSP_LCD_SetFont(&Font16);

    // Initialize layer 1 properties
    SetActiveLayer(Foreground);
    SetBackgroundColor(actualDrawProp[1].BackColor);
    Clear();
    SetForegroundColor(actualDrawProp[1].TextColor);
    BSP_LCD_SetFont(&Font16);

    // Set layers transparency
    SetLayersTransparency(0xFF, 0xFF);
    SetLayersVisibility(true, true);

    _cursorPos[Background].X = 0;
    _cursorPos[Background].Y = 0;
    _cursorPos[Foreground].X = 0;
    _cursorPos[Foreground].Y = 0;
}


RK043FN48H::~RK043FN48H()
{
    BSP_LCD_DeInit();
}


void RK043FN48H::Clear()
{
    BSP_LCD_Clear(actualDrawProp[_selectedLayer].BackColor);
}


void RK043FN48H::Clear(uint32_t color)
{
    BSP_LCD_Clear(color);
}


void RK043FN48H::ClearLayer(Layer layer, uint32_t color)
{
    Layer oldLayer = GetActiveLayer();

    SetActiveLayer(layer);
    Clear(color);
    SetActiveLayer(oldLayer);
}


void RK043FN48H::SetBackgroundColor(uint32_t color)
{
    actualDrawProp[_selectedLayer].BackColor = color;
    BSP_LCD_SetBackColor(actualDrawProp[_selectedLayer].BackColor);
}


void RK043FN48H::SetForegroundColor(uint32_t color)
{
    actualDrawProp[_selectedLayer].TextColor = color;
    BSP_LCD_SetTextColor(actualDrawProp[_selectedLayer].TextColor);
}


void RK043FN48H::SetDrawColor(uint8_t red, uint8_t green, uint8_t blue, uint8_t alpha)
{
    // Calculate display pixel value for selected color
    uint32_t color = red << 16 | green << 8 | blue | alpha << 24;
    SetForegroundColor(color);
}

uint32_t RK043FN48H::GetDrawColor()
{
    return actualDrawProp[_selectedLayer].TextColor;
}

void RK043FN48H::SetClearColor(uint8_t red, uint8_t green, uint8_t blue, uint8_t alpha)
{
    // Calculate display pixel value for selected color
    uint32_t color = red << 16 | green << 8 | blue | alpha << 24;
    SetBackgroundColor(color);
}

uint32_t RK043FN48H::GetClearColor()
{
    return actualDrawProp[_selectedLayer].BackColor;
}

void RK043FN48H::DrawPoint(int posX, int posY, uint32_t colorMask)
{
    if( posX >= 0 && posX < DisplayWidth() && posY >=0 && posY < DisplayHeight()) {
        BSP_LCD_DrawPixel(posX, posY, colorMask);
    }
}


//void RK043FN48H::DrawLine(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2)
//{
//    BSP_LCD_DrawLine(x1, y1, x2, y2);
//}


void RK043FN48H::CopyBitmap(Layer layer, uint8_t * bitmap, uint32_t width, uint32_t height, uint32_t rgbGolorCode)
{
    // Check size
    if(width > DisplayWidth() || height > DisplayHeight())
        return;
    
    ClearLayer(layer, 0x00000000);

    uint16_t maxCol = width / 8;   // 60 columns
    if ((width % 8) != 0)
    {
        maxCol++;
    }
    
    for (int y = 0; y < height; y++) {
        for (int col = 0; col < maxCol; col++) {
            uint8_t shift = bitmap[y*maxCol + col];
            for (int pos = 0; pos < 8; pos++) {
                int x = (col << 3) + pos;   // x = col * 8 + pos
                if (x >= width)
                    break;

                if(shift & 1 << pos)
                {
                    DrawPoint(x, y, rgbGolorCode);
                }
            }
        }
    }
}


uint16_t RK043FN48H::DisplayWidth()
{
    return BSP_LCD_GetXSize();
}


uint16_t RK043FN48H::DisplayHeight()
{
    return BSP_LCD_GetYSize();
}


void RK043FN48H::SetActiveLayer(Layer layer)
{
    _selectedLayer = layer;
    
    BSP_LCD_SelectLayer((uint32_t)layer);
}


Layer RK043FN48H::GetActiveLayer()
{
    return _selectedLayer;
}


void RK043FN48H::SetLayersTransparency( uint8_t background, uint8_t foreground)
{
    BSP_LCD_SetTransparency(0, background);
    BSP_LCD_SetTransparency(1, foreground);
}


void RK043FN48H::SetLayersVisibility( bool background, bool foreground)
{
    BSP_LCD_SetLayerVisible(0, background ? ENABLE : DISABLE);
    BSP_LCD_SetLayerVisible(1, foreground ? ENABLE : DISABLE);
}


//==============================================================================
//
//  Text output
//
//==============================================================================

void RK043FN48H::GotoXY(int x, int y)
{
    _cursorPos[_selectedLayer].X = x;
    _cursorPos[_selectedLayer].Y = y;
}

void RK043FN48H::gotoNewLine(int width, int height)
{
    GotoXY(0, _cursorPos[_selectedLayer].X + height >= DisplayHeight() ? 0 : _cursorPos[_selectedLayer].Y + height);
}

void RK043FN48H::putch(char ch)
{
    int fontHeight = _selectedFont[_selectedLayer].Height();
    int fontWidth = _selectedFont[_selectedLayer].Width();
     
    if(ch == 0x0A)  // '/r'
        return;
        
    if(ch == 0x0D) {    // '/n'
        gotoNewLine(fontWidth, fontHeight);
        return; 
    }
    
    // Check that the character will fit on the screen
    if(_cursorPos[_selectedLayer].X + fontWidth >= DisplayWidth())
    {
        gotoNewLine(fontWidth, fontHeight);
    }
    
    BSP_LCD_DisplayChar(_cursorPos[_selectedLayer].X, _cursorPos[_selectedLayer].Y, ch);
    GotoXY(_cursorPos[_selectedLayer].X + fontWidth, _cursorPos[_selectedLayer].Y);
}


void RK043FN48H::puts(const char * str)
{
    for(int i=0; i < strlen(str); i++)
        putch(*(str+i));
}


void RK043FN48H::DrawText(int posX, int posY, char * str)
{
    // Draw text horizontaly
    int fontWidth = _selectedFont[_selectedLayer].Width();
    int x = posX;
    
    for(int i=0; i < strlen(str); i++)
    {
        BSP_LCD_DisplayChar(x, posY, str[i]);
        x += fontWidth;
    }
}



