/*
    ArduinoGL.h - OpenGL subset for Arduino.
    Created by Fabio de Albuquerque Dela Antonio
    fabio914 at gmail.com
 */

#ifndef ArduinoGL_h
#define ArduinoGL_h

#include "mbed.h"
#include "Commons.h"
#include "Display.h"

#define MAX_VERTICES 24
#define MAX_MATRICES 8

#define DEG2RAD (M_PI/180.0)

#define LOW  0
#define HIGH 1

/**
  * @brief Specifies how objects will be rasterized
  */
typedef enum {
    GL_NONE = 0,
    /// Treats each vertex as a single point. Vertex n defines point n. N points are drawn.
    GL_POINTS,
    /// Draws a single, convex polygon. Vertices 1 through N define this polygon.
    GL_POLYGON,
    /// Draws a connected group of triangles. 
    /// One triangle is defined for each vertex presented after the first two vertices. 
    /// For odd n, vertices n, n + 1 , and n + 2 define triangle n.
    /// For even n, vertices n + 1 , n, and n + 2 define triangle n. N - 2 triangles are drawn.
    GL_TRIANGLE_STRIP
} GLDrawMode;


/**
  * @brief Specify which matrix is the current matrix
  */
typedef enum {
    /// Applies subsequent matrix operations to the projection matrix stack.
    GL_PROJECTION = 0,
    /// Applies subsequent matrix operations to the modelview matrix stack.
    GL_MODELVIEW
} GLMatrixMode;

/* Masks : Indicates the buffers currently enabled for color writing.*/
#define GL_COLOR_BUFFER_BIT 0x1

typedef struct {
    float x, y, z, w;
} GLVertex;

/* Matrices */

/**
  * @brief Specify which matrix is the current matrix
  * @param mode : Specifies which matrix stack is the target for subsequent matrix operations. 
  * Three values are accepted: GL_MODELVIEW, GL_PROJECTION, and GL_TEXTURE. 
  * The initial value is GL_MODELVIEW. Additionally, if the ARB_imaging extension is supported, GL_COLOR is also accepted.
  */
void glMatrixMode(GLMatrixMode mode);
void glMultMatrixf(float * m);


/**
  * @brief Replace the current matrix with the specified matrix
  * @param m : Specifies a pointer to 16 consecutive values, which are used as the elements of a 4 × 4 column-major matrix.
  * Description :
  * glLoadMatrix replaces the current matrix with the one whose elements are specified by m. 
  * The current matrix is the projection matrix, modelview matrix, or texture matrix, 
  * depending on the current matrix mode (see glMatrixMode).
  * The current matrix, M, defines a transformation of coordinates. For instance, 
  * assume M refers to the modelview matrix. 
  * If v = v ⁡ 0 v ⁡ 1 v ⁡ 2 v ⁡ 3 is the set of object coordinates of a vertex, 
  * and m points to an array of 16 single- or double-precision 
  * floating-point values m = m ⁡ 0 m ⁡ 1 ... m ⁡ 15,
  * then the modelview transformation M ⁡ v does the following:
  * M ⁡ v = m ⁡ 0 m ⁡ 4 m ⁡ 8 m ⁡ 12 m ⁡ 1 m ⁡ 5 m ⁡ 9 m ⁡ 13 m ⁡ 2 m ⁡ 6 m ⁡ 10 m ⁡ 14 m ⁡ 3 m ⁡ 7 m ⁡ 11 m ⁡ 15 × v ⁡ 0 v ⁡ 1 v ⁡ 2 v ⁡ 3
  * Projection and texture transformations are similarly defined.
  */
void glLoadMatrixf(float * m);

/**
  * @brief Replace the current matrix with the identity matrix
  * glLoadIdentity replaces the current matrix with the identity matrix. 
  * It is semantically equivalent to calling glLoadMatrix with the identity matrix
  *  1 0 0 0 0 1 0 0 0 0 1 0 0 0 0 1
  */
void glLoadIdentity(void);

void glPushMatrix(void);
void glPopMatrix(void);


/**
  * @brief Multiply the current matrix with an orthographic matrix
  * Description :
  * glOrtho describes a transformation that produces a parallel projection. The current matrix (see glMatrixMode) is multiplied by this matrix and the result replaces the current matrix, as if glMultMatrix were called with the following matrix as its argument:
  * 2 right - left 0 0 t x 0 2 top - bottom 0 t y 0 0 -2 farVal - nearVal t z 0 0 0 1
  * where
  * t x = - right + left right - left
  * t y = - top + bottom top - bottom
  * t z = - farVal + nearVal farVal - nearVal
  * Typically, the matrix mode is GL_PROJECTION, and left bottom - nearVal and right top - nearVal specify the points on the near clipping plane that are mapped to the lower left and upper right corners of the window, respectively, assuming that the eye is located at (0, 0, 0). - farVal specifies the location of the far clipping plane. Both nearVal and farVal can be either positive or negative.
  *
  * Use glPushMatrix and glPopMatrix to save and restore the current matrix stack.
  *
  * @param left, right: Specify the coordinates for the left and right vertical clipping planes.
  * @param bottom, top: Specify the coordinates for the bottom and top horizontal clipping planes.
  * @param nearVal, farVal: Specify the distances to the nearer and farther depth clipping planes. These values are negative if the plane is to be behind the viewer.
  */
void glOrtho(float left, float right, float bottom, float top, float zNear, float zFar);

/**
  * @brief Define a 2D orthographic projection matrix
  * @param left, right : Specify the coordinates for the left and right vertical clipping planes.
  * @param bottom, top : Specify the coordinates for the bottom and top horizontal clipping planes.
  */
void gluOrtho2D(float left, float right, float bottom, float top);

/**
  * @brief multiply the current matrix by a perspective matrix
  *
  * glFrustum describes a perspective matrix that produces a perspective projection. The current matrix (see glMatrixMode) is multiplied by this matrix and the result replaces the current matrix, as if glMultMatrix were called with the following matrix as its argument:
  * 2 ⁢ nearVal right - left 0 A 0 0 2 ⁢ nearVal top - bottom B 0 0 0 C D 0 0 -1 0
  * A = right + left right - left
  * B = top + bottom top - bottom
  * C = - farVal + nearVal farVal - nearVal
  * D = - 2 ⁢ farVal ⁢ nearVal farVal - nearVal
  * Typically, the matrix mode is GL_PROJECTION, and left bottom - nearVal and right top - nearVal specify the points on the near clipping plane that are mapped to the lower left and upper right corners of the window, assuming that the eye is located at (0, 0, 0). - farVal specifies the location of the far clipping plane. Both nearVal and farVal must be positive.
  * 
  * Use glPushMatrix and glPopMatrix to save and restore the current matrix stack.
  *
  * @param left, right: Specify the coordinates for the left and right vertical clipping planes.
  * @param bottom, top: Specify the coordinates for the bottom and top horizontal clipping planes.
  * @param nearVal, farVal: Specify the distances to the nearer and farther depth clipping planes. These values are negative if the plane is to be behind the viewer.
  */
void glFrustum(float left, float right, float bottom, float top, float zNear, float zFar);

/**
  * @brief Set up a perspective projection matrix
  * Description :
  * gluPerspective specifies a viewing frustum into the world coordinate system. In general, the aspect ratio in gluPerspective should match the aspect ratio of the associated viewport. For example, aspect = 2.0 means the viewer's angle of view is twice as wide in x as it is in y. If the viewport is twice as wide as it is tall, it displays the image without distortion.
  *
  * The matrix generated by gluPerspective is multipled by the current matrix, just as if glMultMatrix were called with the generated matrix. To load the perspective matrix onto the current matrix stack instead, precede the call to gluPerspective with a call to glLoadIdentity.
  * Given f defined as follows:
  * f = cotangent ⁡ fovy 2
  * The generated matrix is
  * f aspect 0 0 0 0 f 0 0 0 0 zFar + zNear zNear - zFar 2 × zFar × zNear zNear - zFar 0 0 -1 0
  *
  * @param fovy: Specifies the field of view angle, in degrees, in the y direction.
  * @param aspect: Specifies the aspect ratio that determines the field of view in the x direction. The aspect ratio is the ratio of x (width) to y (height).
  * @param zNear: Specifies the distance from the viewer to the near clipping plane (always positive).
  * @param zFar: Specifies the distance from the viewer to the far clipping plane (always positive).
  */
void gluPerspective(float fovy, float aspect, float zNear, float zFar);

/**
  * @brief Multiply the current matrix by a rotation matrix
  * @param angle : Specifies the angle of rotation, in degrees.
  * @param x, y, z : Specify the x, y, and z coordinates of a vector, respectively.
  */
void glRotatef(float angle, float x, float y, float z);

/**
  * @brief Multiply the current matrix by a translation matrix
  * @param x, y, z : Specify the x, y, and z coordinates of a translation vector.
  */
void glTranslatef(float x, float y, float z);

/**
  * @brief Multiply the current matrix by a general scaling matrix
  * @param x, y, z : Specify the x, y, and z coordinates of a scaling vector.
  */
void glScalef(float x, float y, float z);

/**
  * @brief Define a viewing transformation
  * gluLookAt creates a viewing matrix derived from an eye point, a reference point indicating the center of the scene, and an UP vector.
  *
  * The matrix maps the reference point to the negative z axis and the eye point to the origin.
  * When a typical projection matrix is used, the center of the scene therefore maps to the center of the viewport. 
  * Similarly, the direction described by the UP vector projected 
  * onto the viewing plane is mapped to the positive y axis so that 
  * it points upward in the viewport. 
  * The UP vector must not be parallel to the line of sight from the eye point to the reference point.
  * Let
  * F = centerX - eyeX centerY - eyeY centerZ - eyeZ
  * Let UP be the vector upX upY upZ .
  * 
  * Then normalize as follows:
  * 
  * f = F F
  * UP ″ = UP UP
  * Finally, let s = f × UP ″ , and u = s s × f .
  * 
  * M is then constructed as follows:
  * 
  * M = s ⁡ 0 s ⁡ 1 s ⁡ 2 0 u ⁡ 0 u ⁡ 1 u ⁡ 2 0 - f ⁡ 0 - f ⁡ 1 - f ⁡ 2 0 0 0 0 1
  * and gluLookAt is equivalent to
  * 
  * glMultMatrixf(M);
  * glTranslated(-eyex, -eyey, -eyez);
  *
  * @param eyeX, eyeY, eyeZ : Specifies the position of the eye point.
  * @paramcenterX, centerY, centerZ : Specifies the position of the reference point.
  * @paramupX, upY, upZ: Specifies the direction of the up vector.
  */
void gluLookAt(float eyeX, float eyeY, float eyeZ, float centerX, float centerY, float centerZ, float upX, float upY, float upZ);

/* Vertices functions - specifies a vertex.*/

/**
  * @brief Specifies a vertex
  */
void glVertex4fv(float * v);

/**
  * @brief Specifies a vertex
  * The glVertex function commands are used within glBegin/glEnd pairs 
  * to specify point, line, and polygon vertices. 
  * The current color, normal, and texture coordinates 
  * are associated with the vertex when glVertex is called. 
  * When only x and y are specified, z defaults to 0.0 and w defaults to 1.0. 
  * When x, y, and z are specified, w defaults to 1.0. 
  * Invoking glVertex outside of a glBegin/glEnd pair results in undefined behavior.
  * 
  * @param x: Specifies the x-coordinate of a vertex.
  * @param y: Specifies the y-coordinate of a vertex.
  * @param z: Specifies the z-coordinate of a vertex.
  * @param w: Specifies the w-coordinate of a vertex.
  */
void glVertex4f(float x, float y, float z, float w);

/**
  * @brief Specifies a vertex
  */
void glVertex3fv(float * v);

/**
  * @brief Specifies a vertex
  * The glVertex function commands are used within glBegin/glEnd pairs to specify point, line, and polygon vertices. 
  * The current color, normal, and texture coordinates are associated with the vertex when glVertex is called. 
  * When only x and y are specified, z defaults to 0.0 and w defaults to 1.0. 
  * When x, y, and z are specified, w defaults to 1.0. 
  * Invoking glVertex outside of a glBegin/glEnd pair results in undefined behavior.
  * @param x: Specifies the x-coordinate of a vertex.
  * @param y: Specifies the y-coordinate of a vertex.
  * @param z: Specifies the z-coordinate of a vertex.
  */
void glVertex3f(float x, float y, float z);

/* OpenGL */
void glUseCanvas(Display * c); /* <-- Arduino only */

/**
  * @brief Specify the diameter of rasterized points
  * @param size: Specifies the diameter of rasterized points. The initial value is 1.
  */
void glPointSize(unsigned size);

/**
  * @brief Clear buffers to preset values
  * @param size: Bitwise OR of masks that indicate the buffers to be cleared. 
  * The four masks are GL_COLOR_BUFFER_BIT, GL_DEPTH_BUFFER_BIT, GL_ACCUM_BUFFER_BIT, and GL_STENCIL_BUFFER_BIT.
  */
void glClear(int mask);

/**
  * @brief Delimit the vertices of a primitive or a group of like primitives
  * @param size: Specifies the primitive or primitives that will be created 
  * from vertices presented between glBegin and the subsequent glEnd. 
  * Ten symbolic constants are accepted: 
  * GL_POINTS, GL_LINES, GL_LINE_STRIP, GL_LINE_LOOP, GL_TRIANGLES, 
  * GL_TRIANGLE_STRIP, GL_TRIANGLE_FAN, GL_QUADS, GL_QUAD_STRIP, and GL_POLYGON.
  */
void glBegin(GLDrawMode mode);

/**
  * @brief Delimit the vertices of a primitive or a group of like primitives
  */
void glEnd(void);

#endif