#ifndef AJK_PIN_DETECT_H
#define AJK_PIN_DETECT_H
 
 #ifndef MBED_H
 #include "mbed.h"
 #endif
 
 #ifndef PINDETECT_PIN_ASSTERED
 #define PINDETECT_PIN_ASSTERED   1
 #endif
 
 #ifndef PINDETECT_SAMPLE_PERIOD
 #define PINDETECT_SAMPLE_PERIOD 20000
 #endif
  
 #ifndef PINDETECT_ASSERT_COUNT  
 #define PINDETECT_ASSERT_COUNT  1
 #endif
 
 #ifndef PINDETECT_HOLD_COUNT
 #define PINDETECT_HOLD_COUNT    50
 #endif
 
 namespace AjK {
 
class PinDetect {
 
protected:
DigitalIn   *_in;
     Ticker      *_ticker;
     int         _prevState;
     int         _currentStateCounter;
     int         _sampleTime;
     int         _assertValue;
     int         _samplesTillAssertReload;
     int         _samplesTillAssert;
     int         _samplesTillHeldReload;
     int         _samplesTillHeld;
     FunctionPointer _callbackAsserted;
     FunctionPointer _callbackDeasserted;
     FunctionPointer _callbackAssertedHeld;
     FunctionPointer _callbackDeassertedHeld;
     

     void init(PinName p, PinMode m) {
         _sampleTime              = PINDETECT_SAMPLE_PERIOD;
         _samplesTillAssert       = PINDETECT_ASSERT_COUNT;
         _samplesTillHeld         = 0;
         _samplesTillAssertReload = PINDETECT_ASSERT_COUNT;
         _samplesTillHeldReload   = PINDETECT_HOLD_COUNT;
         _assertValue             = PINDETECT_PIN_ASSTERED;
         
         _in = new DigitalIn( p );
         _in->mode( m );        
         _prevState = _in->read();        
         _ticker = new Ticker;
     }
     
 public:
 
     friend class Ticker;
     
     PinDetect() { error("You must supply a PinName"); }
 
  
   
     PinDetect(PinName p) {
        init( p, PullDown );
     }

 
     PinDetect(PinName p, PinMode m) {
         init( p, m );
     }
     
   
     ~PinDetect() {
         if ( _ticker )  delete( _ticker );
         if ( _in )      delete( _in );
     }
     

     void setSampleFrequency(int i = PINDETECT_SAMPLE_PERIOD) { 
         _sampleTime = i; 
         _prevState  = _in->read();        
         _ticker->attach_us( this, &PinDetect::isr, _sampleTime );
     }
     

     void setAssertValue (int i = PINDETECT_PIN_ASSTERED) { _assertValue = i & 1; }
     
    
     void setSamplesTillAssert(int i) { _samplesTillAssertReload = i; }
     
   
     void setSamplesTillHeld(int i) { _samplesTillHeldReload = i; }
     

     void mode(PinMode m) { _in->mode( m ); }
     

     void attach_asserted(void (*function)(void)) {
         _callbackAsserted.attach( function );
     }
     

     template<typename T>
     void attach_asserted(T *object, void (T::*member)(void)) {
         _callbackAsserted.attach( object, member );        
     }
     

     void attach_deasserted(void (*function)(void)) {
         _callbackDeasserted.attach( function );
     }
     

     template<typename T>
     void attach_deasserted(T *object, void (T::*member)(void)) {
         _callbackDeasserted.attach( object, member );        
     }
     

     void attach_asserted_held(void (*function)(void)) {
         _callbackAssertedHeld.attach( function );
     }
     

     template<typename T>
     void attach_asserted_held(T *object, void (T::*member)(void)) {
         _callbackAssertedHeld.attach( object, member );        
     }
     

     void attach_deasserted_held(void (*function)(void)) {
         _callbackDeassertedHeld.attach( function );
     }
     

     template<typename T>
     void attach_deasserted_held(T *object, void (T::*member)(void)) {
         _callbackDeassertedHeld.attach( object, member );        
     }
     

     operator int() { return _in->read(); }
 
 protected:    

     void isr(void) {
         int currentState = _in->read();
     
         if ( currentState != _prevState ) {
             if ( _samplesTillAssert == 0 ) {
                 _prevState = currentState;
                 _samplesTillHeld = _samplesTillHeldReload;
                 if ( currentState == _assertValue ) 
                     _callbackAsserted.call();
                 else                              
                     _callbackDeasserted.call();
             }
             else {
                 _samplesTillAssert--;
             }
         }
         else {
             _samplesTillAssert = _samplesTillAssertReload;
         }
         
         if ( _samplesTillHeld ) {
             if ( _prevState == currentState ) {
                 _samplesTillHeld--;
                 if ( _samplesTillHeld == 0 ) {
                     if ( currentState == _assertValue ) 
                         _callbackAssertedHeld.call();
                     else                              
                         _callbackDeassertedHeld.call();
                 }
             }
             else {
                 _samplesTillHeld = 0;
             }
         }
     }
     
 };
 
 }
 
 using namespace AjK;
 
 #endif
