// $Id$
// 
// Implements the datagram transport mechanism
//
#include "udp.h"

udp::udp()
:	// default defaults for non-strings
	header_version(0),
	header_size(32),
	message_version(0.0),
	portnum(6267+32768+1), // see wiki docs
	debug_level(0)
{	
	timeout.tv_sec = 1;
	timeout.tv_usec = 0;
	// default defaults for string
	set_message_format("NONE");
	set_hostname("127.0.0.1");
	set_uri("");

#ifdef WIN32
	// initialize socket subsystem
	WORD wsaVersion = MAKEWORD(2,2);
	WSADATA wsaData;
	if ( WSAStartup(wsaVersion,&wsaData)!=0 )
	{
		set_errormsg("unable to initialize Windows sockets");
		exception("udp::udp() failed");
	}
#endif
	// default socket is non-existent
	sd = INVALID_SOCKET;

	// allocate needed buffers
	sockdata = new struct sockaddr_in;

	// reset the error message to anticipate create coming next
	set_errormsg("udp::create() not called");
}

udp::~udp()
{
	flush();
}

/// udp pseudo-property handler
int udp::option(char *command)
{
	char param[256], value[1024];
	while ( command!=NULL && *command!='\0' ) 
	{
		switch ( sscanf(command,"%256[^ =]%*[ =]%[^,;]",param,value) ) {
		case 1:
			// TODO
			error("option \"transport:%s\" not recognized", command);
			return 0;
		case 2:
			if ( strcmp(param,"port")==0 )
				set_portnum(atoi(value));
			else if ( strcmp(param,"header_version")==0 )
				set_header_version(atoi(value));
			else if ( strcmp(param,"timeout")==0 )
				set_timeout(atoi(value));
			else if ( strcmp(param,"hostname")==0 )
				set_hostname(value);
			else if ( strcmp(param,"uri")==0 )
				set_uri("%s",value);
			else if ( strcmp(param,"debug_level")==0 )
				set_debug_level(atoi(value));
			else if ( strcmp(param,"on_error")==0 )
			{
				if ( strcmp(value,"abort")==0 )
					onerror_abort();
				else if ( strcmp(value,"retry")==0 )
					onerror_retry();
				else if ( strcmp(value,"ignore")==0 )
					onerror_ignore();
				else
				{
					error("option \"transport:%s\" not a valid on_error command", command);
					return 0;
				}
			}
			else if ( strcmp(param,"maxretry")==0 )
			{
				int n = atoi(value);
				if ( strcmp(value,"none")==0 )
					set_maxretry();
				else 
				{
					if ( n>0 )
						set_maxretry(n);
					else
					{
						error("option \"transport:%s\" not a valid maxretry command", command);
						return 0;
					}
				}
			}
			else
			{	
				error("option \"transport:%s\" not recognized", command);
				return 0;
			}
			break;
		default:
			error("option \"transport:%s\" cannot be parsed", command);
			return 0;
		}
		char *comma = strchr(command,',');
		char *semic = strchr(command,';');
		if ( comma && semic )
			command = min(comma,semic);
		else if ( comma )
			command = comma;
		else if ( semic )
			command = semic;
		else
			command = NULL;
		if ( command )
		{
			while ( isspace(*command) || *command==',' || *command==';' ) command++;
		}
	}
	return 1;
}

int udp::create(void) 
{
	// create socket
	sd = socket(AF_INET,SOCK_DGRAM,0);
	if ( sd==INVALID_SOCKET ) 
	{
		set_errormsg("udp::create() is unable to create a socket: %s",Socket::strerror());
		return 0;
	}

	// erase the sockdata structure
	set_sockdata(NULL);

	// reset the error message to anticipate init coming next
	set_errormsg("udp::init() not called");
	return 1; /* return 1 on success, 0 on failure */
}

int udp::init(void)
{
	// set up server address info
	struct sockaddr_in &serv_addr = *get_sockdata();
	serv_addr.sin_family = AF_INET;

	// lookup server
	debug(0, "looking up '%s:%d'", hostname, portnum); 
	struct hostent *server = gethostbyname(hostname);
	if ( server )
	{
		debug(1,"IP address for '%s' ok", hostname);
		memcpy((char*)&serv_addr.sin_addr.s_addr,(char*)server->h_addr,server->h_length);
		serv_addr.sin_port = htons(portnum);
	}
	else
		exception("IP address for '%s' cannot be resolved: ", hostname, Socket::strerror());

	// connect to server
	unsigned int retries = 3;
Retry:
	if ( ::connect(sd,(struct sockaddr*)&serv_addr,sizeof(serv_addr)) < 0 )
	{
		if ( retries-->0 ) goto Retry;
		error("unable to connect to server '%s' on port %d: %s", hostname, portnum, Socket::strerror());
		return 0;
	}
	return 1;
}

size_t udp::send(const char *msg, size_t len)
{
	if ( msg==NULL )
	{
		msg = output;
		len = position;
	}
	// format outbound message header
	char temp[256];
	int tlim = (int)ceil((double)timeout.tv_usec/1000.0) + (int)timeout.tv_sec;
	if ( tlim>0 ) tlim=9; else if ( tlim<1 ) tlim=1;
	sprintf(temp,"%-1d %-3d %-7d %-5.5s %-3.1f %-1d %-3d   ", 
		header_version, header_size, len, message_format, message_version, tlim, 0);
	if ( len>1500-strlen(temp) )
	{
		error("udp::send(const char *msg='%-10.10s', size_t len=%d): message is too long for UDP", msg, len);
		return 0;
	}
	char sendbuf[2048];
	int totlen = sprintf(sendbuf,"%s%s",temp,msg);
	struct sockaddr_in &serv_addr = *(struct sockaddr_in *)sockdata;
	size_t sndlen = sendto(sd,sendbuf,totlen,0,(struct sockaddr*)&serv_addr,sizeof(serv_addr));
	debug(9,"%d <= sendto(addr='%s',port=%d,msg='%s')", sndlen, inet_ntoa(serv_addr.sin_addr), ntohs(serv_addr.sin_port), sendbuf);
	if ( sndlen==SOCKET_ERROR )
		exception("UDP sendto failed: %s", Socket::strerror());
	return sndlen;
}
size_t udp::recv(char *buf, size_t len)
{
	if ( buf==NULL )
	{
		memset(input,0,sizeof(input));
		buf = input;
		len = sizeof(input);
	}
	char msg[2048]; // datagrams cannot exceed metric anyway
	struct sockaddr_in &serv_addr = *(struct sockaddr_in *)sockdata;
	struct sockaddr_in remote;
	socklen_t remote_len = sizeof(remote);

	if ( call_setsockopt(sd, SOL_SOCKET, SO_RCVTIMEO, &this->timeout, sizeof(this->timeout))!=0 )
		exception("unable to set timeout for recv_udp"); // new line of code from Dave to fix timeout
	memset(msg,0,sizeof(msg));
	int retry = maxretry;
Retry:
	size_t rcvlen = recvfrom(sd,msg,2047,0,(struct sockaddr*)&remote,&remote_len);
	debug(9,"%d <= recvfrom(addr='%s',port=%d,msg='%s')", rcvlen, inet_ntoa(serv_addr.sin_addr), ntohs(serv_addr.sin_port), msg);
	if ( rcvlen==SOCKET_ERROR )
	{
		switch ( on_error ) {
		case TE_RETRY:
			if ( maxretry==-1 || retry-->0 )
			{
				if ( retry<0 )
					debug(9,"udp::recv() socket error, retrying (no maxretry)");
				else
					debug(9,"udp::recv() socket error, %d retries left", retry);
				goto Retry;
			}
			// fall through to abort
		case TE_ABORT:
			debug(9,"udp::recv() socket error, aborting");
			return -1; 
		case TE_IGNORE:
			debug(9,"udp::recv() socket error, ignoring");
			return 0;
			break;
		default:
			exception("invalid on_error type");
		}
	}
	msg[rcvlen]='\0';
	if ( rcvlen>0 && rcvlen<sizeof(msg) )
		debug(9,"incoming UDP message from %s:%d = '%s'\n\n", inet_ntoa(remote.sin_addr), ntohs(remote.sin_port), msg);
	else 
		exception("unexpected recvfrom return value: %lld", (int64)rcvlen);
	// check response header
	int version;
	int offset;
	int size;
	char type[256];
	int major, minor;
	int time_out;
	int status;
	
	if ( sscanf(msg,"%d %d %d %6s %d.%d %d %d", 
		&version, &offset, &size, type, &major, &minor, &time_out, &status)<8 )
		exception("incomplete or invalid message header {msg='%s'}", msg);

	// check version 
	if ( version!=get_header_version() )
		exception("incorrect header version {msg='%s'}, expected version=%d", msg, get_header_version());

	// check offset
	if ( offset!=get_header_size() )
		exception("unexpected header size {msg='%s'}, expected size=%d", msg, get_header_size());

	// check type
	if ( strcmp(type,get_message_format())!=0 )
		exception("unexpected message format {msg='%s'}, expected format='%s'", msg, get_message_format());

	// check type version (allow diffs up to 0.1)
	if ( fabs((double)major+((double)minor)/10 - get_message_version())>0.099 )
		exception("unexpected message version {msg='%s'}, expected version=%.1f", msg, get_message_version());

	// get body
	if ( size>=rcvlen-1 )
		exception("response too long for buffer (content-length=%d, buffer-size=%d)--truncating message", size,len);
	
	// check code
	if ( status!=200 )
		exception("unexpected response code {msg='%s'}, expected code=200", msg);

	// copy result
	memcpy(buf,msg+offset,size);
	buf[size] = '\0';
	debug(2,"udp::recv() => [%s]",buf);

	// destroy the existing translation (if any)
	if ( translator!=NULL )
		translation = (*translator)(buf,translation);

	return size;
}
int udp::call_setsockopt(SOCKET s, int level, int optname, timeval *optval, int optlen)
{
#ifdef WIN32
	int time_out;
	time_out = (int)optval->tv_usec + (int)(optval->tv_sec*1000);
	return setsockopt(s, level, optname, (const char*)&time_out, sizeof(time_out));
#else
	return setsockopt(s, level, optname, optval, optlen);
#endif
}
void udp::flush(void)
{
}

// read accessors
unsigned int udp::get_header_version(void) const
{
	return header_version;
}
unsigned int udp::get_header_size(void) const
{
	return header_size;
}
const char *udp::get_message_format(void) const
{
	return message_format;
}
double udp::get_message_version(void) const
{
	return message_version;
}
timeval udp::get_timeout(void) const
{
	return timeout;
}
const char *udp::get_hostname(void) const
{
	return hostname;
}
unsigned int udp::get_portnum(void) const
{
	return portnum;
}
const char *udp::get_uri(void) const
{
	return uri;
}
const char *udp::get_errormsg(void) const
{
	return errormsg;
}
unsigned int udp::get_debug_level(void) const
{
	return debug_level;
}
struct sockaddr_in *udp::get_sockdata(size_t n) const
{
	return (struct sockaddr_in*)sockdata;
}

// write accessors
void udp::set_header_version(unsigned int n)
{
	header_version = n;
}
void udp::set_header_size(unsigned int n)
{
	header_size = n;
}
void udp::set_message_format(char *s)
{
	strcpy(message_format,s);
}
void udp::set_message_version(double x)
{
	message_version = x;
}
void udp::set_timeout(unsigned int n)
{
	if(n > 1000){
		timeout.tv_sec = n/1000;
		timeout.tv_usec = n % 1000;
	} else {
		timeout.tv_sec = 0;
		timeout.tv_usec = n;
	}
}
void udp::set_hostname(char *s)
{
	strcpy(hostname,s);
}
void udp::set_portnum(unsigned int n)
{
	portnum = n;
}
void udp::set_uri(char *fmt, ...)
{
	va_list ptr;
	va_start(ptr,fmt);
	vsprintf(uri,fmt,ptr);
	va_end(ptr);
}
void udp::set_errormsg(char *fmt, ...)
{
	va_list ptr;
	va_start(ptr,fmt);
	vsprintf(errormsg,fmt,ptr);
	va_end(ptr);
}
void udp::set_debug_level(unsigned int n)
{
	debug_level = n;
}
void udp::set_sockdata(struct sockaddr_in *p, size_t n)
{
	// TODO
}
void udp::set_output(char *fmt, ...)
{
	va_list ptr;
	va_start(ptr,fmt);
	position = vsprintf(output,fmt,ptr);
	va_end(ptr);
}
void udp::add_output(char *fmt, ...)
{
	va_list ptr;
	va_start(ptr,fmt);
	position += vsprintf(output+position,fmt,ptr);
	va_end(ptr);
}
