/*
 * Reading built-in temperature sensor of STM32F103RB chip (on a NUCLEO-F103RB board)
 */

#include "mbed.h"
#include "LIS3DH.h"


#define MOSI PA_7
#define MISO PA_6
#define CS PA_4
#define SCLK PA_5




LIS3DH      acc(MOSI, MISO, SCLK, CS, LIS3DH_DR_NR_LP_50HZ, LIS3DH_FS_2G);

/*
 * STM32F103x data-sheet:
 * 5.3.19 Temperature sensor characteristics
 * Table 50. TS characteristics, Page 80
 */
const float         AVG_SLOPE   = 10E-03;      // slope (gradient) of temperature line function  [V/°C]
const float         V25         = 0.75;         // sensor's output voltage at 25°C [V]
const float         ADC_TO_VOLT = 3.3 / 4096;   // conversion coefficient of digital value to voltage [V]
// when using 3.3V ref. voltage at 12-bit resolution (2^12 = 4096)



float accX = 0;
float accY = 0;
float accZ = 0;




Serial              pc(PA_9, PA_10);
DigitalOut          led(PC_13);
ADC_HandleTypeDef   hadc1;                      // ADC handle
uint16_t            adcValue;                   // digital value of sensor
float               vSense;                     // sensor's output voltage [V]
float               temp;                       // sensor's temperature [°C]

/* ADC1 init function */
void MX_ADC1_Init(void)
{
    pc.baud(115200);
    pc.format(8,SerialBase::Odd,1);
    ADC_ChannelConfTypeDef  sConfig;

    /**Common config
        */
    hadc1.Instance = ADC1;
    hadc1.Init.ScanConvMode = ADC_SCAN_DISABLE;
    hadc1.Init.ContinuousConvMode = DISABLE;
    hadc1.Init.DiscontinuousConvMode = DISABLE;
    hadc1.Init.ExternalTrigConv = ADC_SOFTWARE_START;
    hadc1.Init.DataAlign = ADC_DATAALIGN_RIGHT;
    hadc1.Init.NbrOfConversion = 1;
    HAL_ADC_Init(&hadc1);

    /**Configure Regular Channel
        */
    sConfig.Channel = 9;
    sConfig.Rank = 1;
    sConfig.SamplingTime = ADC_SAMPLETIME_239CYCLES_5;
    HAL_ADC_ConfigChannel(&hadc1, &sConfig);
}



// When characteristic input changing
void Accupdate()
{

    accX = float(short((acc.read_reg(LIS3DH_OUT_X_H) << 8) | acc.read_reg(LIS3DH_OUT_X_L))) * 0.001F / 15;
    accY = float(short((acc.read_reg(LIS3DH_OUT_Y_H) << 8) | acc.read_reg(LIS3DH_OUT_Y_L))) * 0.001F / 15;
    accZ = float(short((acc.read_reg(LIS3DH_OUT_Z_H) << 8) | acc.read_reg(LIS3DH_OUT_Z_L))) * 0.001F / 15;


    pc.printf("X acceleration = ");
    pc.printf("%5.2f",accX);
    pc.printf("Y acceleration = ");
    pc.printf("%5.2f",accY);
    pc.printf("Z acceleration = ");
    pc.printf("%5.2f",accZ);
}


int main()
{
    MX_ADC1_Init();                                                 // initialize AD convertor
    while(HAL_ADCEx_Calibration_Start(&hadc1) != HAL_OK);           // calibrate AD convertor

    while(1) {
        HAL_ADC_Start(&hadc1);                                      // start analog to digital conversion
        while(HAL_ADC_PollForConversion(&hadc1, 1000000) != HAL_OK);// wait for completing the conversion
        adcValue = HAL_ADC_GetValue(&hadc1);                        // read sensor's digital value
        vSense = adcValue * ADC_TO_VOLT;                            // convert sensor's digital value to voltage [V]
        /*
         * STM32F103xx Reference Manual:
         * 11.10 Temperature sensor
         * Reading the temperature, Page 235
         * Temperature (in °C) = {(V25 - Vsense) / Avg_Slope} + 25
         */
        temp = (vSense - V25) / AVG_SLOPE + 25.0f;                  // convert sensor's output voltage to temperature [°C]
        pc.printf("temp = %3.1f%cC, voltage = %f\n", temp, 176, vSense);                  // display chip's temperature [°C]
        Accupdate();
        wait_ms(1000);
    }
}
