/*

VFD Control class for GP1059A01

===============
GP1059A01 INTERFACE CONNECTION
---------------
1  D0   2  D1
3  D2   4  D3
5  D4   6  D5
7  D6   8  D7
9  GND  10 INT
11 WR   12 RD
13 CS   14 C/D
15 5V   16 5V
17 5V   18 GND
19 GND  20 ebb(+70V)

=====================
mbed CONNECTION
---------------------
mbed   GP1059A01(I/O)
---------------------
p5     1 (D0)  I/O
p6     2 (D1)  I/O
p7     3 (D2)  I/O
p8     4 (D3)  I/O
p9     5 (D4)  I/O
p10    6 (D5)  I/O
p11    7 (D6)  I/O
p12    8 (D7)  I/O
p13    10 (INT) O
p14    11 (WR)  I
p15    12 (RD)  I
p16    13 (CS)  I
p17    14 (C/D) I
GND    9,18,19 (GND)
=====================

*/

#ifndef __VFD_GP1059_H__
#define __VFD_GP1059_H__

#include "mbed.h"

class VFD_GP1059 {
private:
    BusOut data;
    DigitalOut wr,rd,cs,cd;
    DigitalIn intr;

    void init(){
        cs = 1;
        wr = 1;
        rd = 1;
        cd = 1;
    }

public:
    // constructor
    VFD_GP1059(PinName d0_pin,
               PinName d1_pin,
               PinName d2_pin,
               PinName d3_pin,
               PinName d4_pin,
               PinName d5_pin,
               PinName d6_pin,
               PinName d7_pin,
               PinName intr_pin,
               PinName wr_pin,
               PinName rd_pin,
               PinName cs_pin,
               PinName cd_pin) :
               data(d0_pin, d1_pin, d2_pin, d3_pin, d4_pin, d5_pin, d6_pin, d7_pin),
               intr(intr_pin),
               wr(wr_pin),
               rd(rd_pin),
               cs(cs_pin),
               cd(cd_pin) {        
        init();
        cls();
    }

    void send_cmd(uint8_t cmd){
        cd = 1;       // C/D SET HIGH      
        data = cmd;   // COMMAND SET
        cs = 0;       // CS SET LOW
        wr = 0;       // WR SET LOW
        wait_us(2);   // wait 2us
        wr = 1;       // WR SET HIGH
        cs = 1;       // CS SET HIGH
        wait_us(4);   // wait 4us

        return;
    }

    void send_data(uint8_t data_value){
        cd = 0;       // C/D SET HIGH
        data = data_value;   // DATA SET
        cs = 0;       // CS SET LOW
        wr = 0;       // WR SET LOW
        wait_us(2);   // wait 2us
        wr = 1;       // WR SET HIGH
        cs = 1;       // CS SET HIGH
        wait_us(4);   // wait 4us
    
        return;
    }

    // Luminance Adjustment (06H)
    void luminance_adjustment(uint8_t data){
        send_cmd(6);
        send_data(data);

        return;
    }    

    // 04H,05H,02H: Setting address of Write
    void set_write_read_address(uint16_t address){
        send_cmd(4);    // 04H: Setting lower address of Write-Read
        send_data((uint8_t)(address & 0x0ff));    // mask upper address 
    
        send_cmd(5);    // 05H: Setting upper address of Write-Read
        send_data((uint8_t)(address >> 8));       // 8bit shift

        send_cmd(2);    // 02H: Data Write
    
        return;
    }

    // 07H,08H: Setting address display started
    void set_disp_start_address(uint16_t address){
        send_cmd(7);    // 07H: Setting lower address display started
        send_data((uint8_t)(address & 0x0ff));
    
        send_cmd(8);    // 08H: Setting upper address display started
        send_data((uint8_t)(address >> 8));
    
        return;
    }
    
    // Clear display memory
    void cls(){
        set_disp_start_address(0);
        set_write_read_address(0);
        for (int i = 0; i < 0x1fff; i++){
             send_data(0);
        }
    return;
    }

};

#endif