/*

mbed VFD Control for GP1059A01

===============
Status LED
-----------
LED1: 
LED2: 
LED3: 
LED4: 

===============
GP1059A01 INTERFACE CONNECTION
---------------
1  D0   2  D1
3  D2   4  D3
5  D4   6  D5
7  D6   8  D7
9  GND  10 INT
11 WR   12 RD
13 CS   14 C/D
15 5V   16 5V
17 5V   18 GND
19 GND  20 ebb(+70V)



=====================
mbed CONNECTION
---------------------
mbed   GP1059A01(I/O)
---------------------
p5     1 (D0)  I/O
p6     2 (D1)  I/O
p7     3 (D2)  I/O
p8     4 (D3)  I/O
p9     5 (D4)  I/O
p10    6 (D5)  I/O
p11    7 (D6)  I/O
p12    8 (D7)  I/O
p13    10 (INT) O
p14    11 (WR)  I
p15    12 (RD)  I
p16    13 (CS)  I
p17    14 (C/D) I
GND    9,18,19 (GND)
=====================

*/

#include "mbed.h"

Serial pc(USBTX, USBRX);
LocalFileSystem local("local");

// VFD DATA PORT
BusOut DATA(p5, p6, p7, p8, p9, p10, p11, p12);
//               __
// Write Signal (WR)
DigitalOut WR(p14);
//              __
// Read Signal (RD)
DigitalOut RD(p15);
//                     __
// Chip Select Signal (CS)
DigitalOut CS(p16);
//                                 _
// Command / Data Select Signal (C/D)
DigitalOut CD(p17);
//
// Frame Signal
DigitalIn INT(p13);

// LED Status
DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut led4(LED4);


const uint8_t HIGH = 1;
const uint8_t LOW = 0;
const uint16_t MAX_ADDRESS = 0x01fff;

FILE *fp;
char in_filename[] = "/local/rom1.bin";

void send_cmd(uint8_t cmd)
{
    CD = HIGH;    // C/D SET HIGH      
    DATA = cmd;   // COMMAND SET
    CS = LOW;     // CS SET LOW
    WR = LOW;     // WR SET LOW
    wait_us(2);   // wait 2us
    WR = HIGH;    // WR SET HIGH
    CS = HIGH;    // CS SET HIGH
    wait_us(4);   // wait 4us

    return;
}

void send_data(uint8_t data)
{
    CD = LOW;      // C/D SET HIGH
    DATA = data;   // DATA SET
    CS = LOW;      // CS SET LOW
    WR = LOW;      // WR SET LOW
    wait_us(2);    // wait 2us
    WR = HIGH;     // WR SET HIGH
    CS = HIGH;     // CS SET HIGH
    wait_us(4);    // wait 4us
    
    return;
}

// Luminance Adjustment (06H)
void luminance_adjustment(uint8_t data)
{
    send_cmd(6);
    send_data(data);

    return;
}    

// 04H,05H,02H: Setting address of Write
void set_write_read_address(uint16_t address){
    send_cmd(4);    // 04H: Setting lower address of Write-Read
    send_data((uint8_t)(address & 0x0ff));    // mask upper address 
    
    send_cmd(5);    // 05H: Setting upper address of Write-Read
    send_data((uint8_t)(address >> 8));       // 8bit shift

    send_cmd(2);    // 02H: Data Write
    
    return;
}

// 07H,08H: Setting address display started
void set_disp_start_address(uint16_t address){
    send_cmd(7);    // 07H: Setting lower address display started
    send_data((uint8_t)(address & 0x0ff));
    
    send_cmd(8);    // 08H: Setting upper address display started
    send_data((uint8_t)(address >> 8));
    
    return;
}

// TEST PATTERN
void test_pattern(){
    led1 = 1;
    set_disp_start_address(0);  // Display address: 000000H
    led1 = 0;
    
    led2 = 1;
    luminance_adjustment(0x0f);    // 100%
    led2 = 0;
    
    for (uint16_t d = 0; d < 256 ; d++){
        led3 = 1;
        set_write_read_address(0);  // Start address: 000000H
        led3 = 0;
    
        for (uint16_t i = 0; i < MAX_ADDRESS; i++){
            led4 = 1;
            send_data(d);
            led4 = 0;
        }
    }
    
    return;        
}

void test_datafile(){
    if ( NULL == (fp = fopen( in_filename, "r" )) ) {
        pc.printf("File open error.\n");
        led1 = HIGH;
        return;
    }

    led1 = 1;
    set_disp_start_address(0);  // Display address: 000000H
    led1 = 0;
    
    led2 = 1;
    luminance_adjustment(0x0f); // 100%
    led2 = 0;
    
    led3 = 1;
    set_write_read_address(0);  // Start address: 000000H
    led3 = 0;
    
    char d[8];  // 1char 8byte buffer
    int char_cnt = 0;
    
    // 8byte = 1CHAR
    while(fgets(d, 8, fp) != NULL && char_cnt < 10) {
        for (int byte_num = 0; byte_num < 8 ; byte_num++){ // byte
            pc.printf("d[%d]:%02x ",byte_num,d[byte_num]);
            uint8_t a = 0;
            for (int bit_num = 7; bit_num >= 0; bit_num--){ // bit
                if ((d[byte_num] & 0x80) != 0) {
                    a = a + (2 ^ bit_num);
                }
                d[byte_num] = d[byte_num] << 1;
                pc.printf("a:%02x d:%02x ",a,d[byte_num]);
            }
            if ((d[byte_num] & 0x80) != 0) a++;
            
            led4 = 1;
            pc.printf("A:%02x ",a);
            send_data(a);
            led4 = 0;

            for (int e = 0; e < 7 ; e++) send_data(0);
        }
        char_cnt++;
    }
    fclose(fp);

    return;        
}

void blink_LED() {
    for (int i = 0 ; i < 5 ; i++) {
        led4 = 1;
        wait(0.2);
        led4 = 0;
        wait(0.2);
    }
    return;
}

int main() {

    led1 = LOW;
    led2 = LOW;
    led3 = LOW;
    led4 = LOW;
    
    CS = HIGH;
    WR = HIGH;
    RD = HIGH;
    CD = HIGH;
    
    // Start Sign
    blink_LED();
    
    // TEST
    //test_pattern();
    
    test_datafile();
    
    // END Sign
    blink_LED();    
}