#ifndef MBED_AKILEDMATRIX_H
#define MBED_AKILEDMATRIX_H
 
#include "mbed.h"

/** AkiLedMatrix class.
 *  Used for AKIZUKI MATRIX LED unit.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "akiledmatrix.h"
 *
 * // LEDMatrix      mbed
 * // ----------   --------
 * // 1 LED_PWR
 * // 2 SIN_1 ----- p5  line
 * // 3 SIN_2 ----- p6  LED1 row
 * // 4 SIN_3 ----- p7  LED2 row
 * // 5 CLOCK ----- p8
 * // 6 LATCH ----- p9
 * // 7 STROBE ---- p10
 * // 8 IC_PWR
 * // 9 GND
 * // 10 GND
 *
 * // ledunit = 1 units
 * // rowsize = 4 bytes
 * // dynamic_delay = 1000 us
 * // scroll_delay = 50 count
 *
 * AkiLedMatrix ledmatrix(p5, p6, p7, p8, p9, p10, 1, 4, 1000, 50);
 * 
 * int main() {
 *    const unsigned char buf[] = {
 *    0xff,0xff,0xff,0xff,  // D15   
 *    0xff,0xff,0xff,0xff,  // D14
 *    0xff,0xff,0xff,0xff,  // D13
 *    0xff,0xe7,0xff,0xfc,  // D12
 *    0xff,0xe7,0xff,0xfc,  // D11
 *    0xff,0xe7,0xff,0xfc,  // D10
 *    0x44,0x60,0xe0,0xe0,  // D9
 *    0x00,0x20,0x40,0x40,  // D8
 *    0x33,0x26,0x4e,0x4c,  // D7
 *    0x33,0x26,0x40,0x4c,  // D6
 *    0x33,0x26,0x4f,0xcc,  // D5
 *    0x33,0x20,0x60,0xc0,  // D4
 *    0x33,0x28,0xf1,0xe4,  // D3
 *    0xff,0xff,0xff,0xff,  // D2
 *    0xff,0xff,0xff,0xff,  // D1
 *    0xff,0xff,0xff,0xff}; // D0
 * 
 *    while(1){
 *        ledmatrix.display(buf);
 *    }
 * }
 * @endcode
 */

class AkiLedMatrix {
public:
    /** Create AkiLedMatrix instance
     * @param sin1 SIN_1
     * @param sin2 SIN_2
     * @param sin3 SIN_3
     * @param clock CLOCK IN
     * @param latch LATCH IN
     * @param strobe STROBE IN
     * @param ledunit Number of LED units
     * @param rowsize LED buffer row size
     * @param delay Dynamic display delay in microseconds
     * @param shift_count_init scroll timing. if 0 is No scroll.
     */
    AkiLedMatrix(PinName sin1,
                 PinName sin2,
                 PinName sin3,
                 PinName clock,
                 PinName latch,
                 PinName strobe,
                 const int ledunit,
                 const int rowsize,
                 const int delay,
                 const int shift_count_init);
                                  
    /** Displays the contents of the buffer
     *
     * @param buffer Display buffer
     * @returns
     *        void
     */
    void display(unsigned char *buffer);

    /** Get number of LED units connecting
     *
     * @param
     *        void
     * @returns
     *        number of LED units
     */
    int getledunit();

    /** Get LED buffer row size
     *
     * @param
     *        void
     * @returns
     *        LED buffer row size
     */
    int getrowsize();
 
private:
    /** bit left shift
     *
     * @param
     *        unsigned char *array LED display buffer
     *        int xsize row char size
     * @returns
     *        void
     */
    void bitshift(unsigned char *array, int xsize);

    DigitalOut _sin1,_sin2,_sin3,_clock,_latch,_strobe;
    const int _ledunit;
    const int _rowsize;
    const int _delay;
    const int _shift_count_init;
    int shift_count;
};
 
#endif
