/* MCP4922 Driver Library
 * Copyright (c) 2014 Neil Thiessen
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef MCP4922_H
#define MCP4922_H

#include "mbed.h"

/** MCP4922 class.
 *  Used for controlling an MCP4922 DAC connected via SPI.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "MCP4922.h"
 *
 * //Create an MCP4922 object
 * MCP4922 dac(p11, p13, p14);
 *
 * int main()
 * {
 *     //Configure DAC A
 *     dac.referenceMode(MCP4922::DAC_A, MCP4922::REF_UNBUFFERED);
 *     dac.gainMode(MCP4922::DAC_A, MCP4922::GAIN_1X);
 *     dac.powerMode(MCP4922::DAC_A, MCP4922::POWER_NORMAL);
 *
 *     while (1) {
 *         //Generate a sine wave on DAC A
 *         for (float i = 0.0; i < 360.0; i += 0.1)
 *             dac.write(MCP4922::DAC_A, 0.5 * (sinf(i * 3.14159265 / 180.0) + 1));
 *     }
 * }
 * @endcode
 */
class MCP4922
{
public:
    /** Represents the different DACs in the MCP4922
     */
    enum DAC {
        DAC_A,              /**< DAC A */
        DAC_B               /**< DAC B */
    };

    /** Represents the reference buffer mode of DAC A or B in the MCP4922
     */
    enum ReferenceMode {
        REF_UNBUFFERED,     /**< DAC VREF is unbuffered */
        REF_BUFFERED        /**< DAC VREF is buffered */
    };

    /** Represents the gain mode of DAC A or B in the MCP4922
     */
    enum GainMode {
        GAIN_2X,            /**< DAC output = 2 * VREF * D/4096 */
        GAIN_1X             /**< DAC output = VREF * D/4096 */
    };

    /** Represents the power mode of DAC A or B in the MCP4922
     */
    enum PowerMode {
        POWER_SHUTDOWN,     /**< DAC output is high impedance */
        POWER_NORMAL        /**< DAC output is enabled */
    };

    /** Create an MCP4922 object connected to the specified SPI pins
     *
     * @param mosi The SPI data out pin.
     * @param sclk The SPI clock pin.
     * @param cs The SPI chip select pin.
     * @param hz The SPI bus frequency (defaults to 20MHz).
     */
    MCP4922(PinName mosi, PinName miso, PinName sclk, PinName cs, int hz = 20000000);

    /** Get the current reference mode of the specified DAC in the MCP4922
     *
     * @param dac The DAC to read from.
     *
     * @returns The current reference mode of the specified DAC as a ReferenceMode enum.
     */
    MCP4922::ReferenceMode referenceMode(DAC dac);

    /** Set the reference mode of the specified DAC in the MCP4922
     *
     * @param dac The DAC to write to.
     * @param mode The new reference mode for the specified DAC as a ReferenceMode enum.
     */
    void referenceMode(DAC dac, ReferenceMode mode);

    /** Get the current gain mode of the specified DAC in the MCP4922
     *
     * @param dac The DAC to read from.
     *
     * @returns The current gain mode of the specified DAC as a GainMode enum.
     */
    MCP4922::GainMode gainMode(DAC dac);

    /** Set the gain mode of the specified DAC in the MCP4922
     *
     * @param dac The DAC to write to.
     * @param mode The new gain mode for the specified DAC as a GainMode enum.
     */
    void gainMode(DAC dac, GainMode mode);

    /** Get the current power mode of the specified DAC in the MCP4922
     *
     * @param dac The DAC to read from.
     *
     * @returns The current power mode of the specified DAC as a PowerMode enum.
     */
    MCP4922::PowerMode powerMode(DAC dac);

    /** Set the power mode of the specified DAC in the MCP4922
     *
     * @param dac The DAC to write to.
     * @param mode The new power mode for the specified DAC as a PowerMode enum.
     */
    void powerMode(DAC dac, PowerMode mode);

    /** Get the current output voltage of the specified DAC in the MCP4922 as a percentage
     *
     * @param dac The DAC to read from.
     *
     * @returns The current output voltage of the specified DAC as a percentage (0.0 to 1.0 * VDD).
     */
    float read(DAC dac);
    
    uint16_t read_u16(DAC dac);

    /** Set the output voltage of the specified DAC in the MCP4922 from a percentage
     *
     * @param dac The DAC to write to.
     * @param value The new output voltage for the specified DAC as a percentage (0.0 to 1.0 * VDD).
     */
    void write(DAC dac, float value);

    /** Set the output voltage of the specified DAC in the MCP4922 from a 16-bit range
     *
     * @param dac The DAC to write to.
     * @param value The new output voltage for the specified DAC as a 16-bit unsigned short (0x0000 to 0xFFFF).
     */
    void write_u16(DAC dac, uint16_t value);
    
    // focntion directte sur les bits : 
    void write_bits(DAC dac, uint16_t value);
    
    // DAC A : 
    void write_DAC_A(uint16_t value);
    
    // DAC B : 
    void write_DAC_B(uint16_t value);

private:
    //SPI member variables
    SPI m_SPI;
    DigitalOut m_CS;

    //DAC settings member variables
    unsigned short m_DacValueA;
    unsigned short m_DacValueB;

    //Internal functions
    void writeDac(unsigned short value);
};

#endif






#if 0 

/* mbed MCP4822 Library, for driving the 12-Bit DAC with internal Vref and SPI interface
 * Copyright (c) 2015, Created by Steen Joergensen (stjo2809)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#include "mbed.h"

#ifndef MBED_MCP4822_H
#define MBED_MCP4822_H

//=============================================================================
// Declaration of variables & custom #defines
//=============================================================================

#define CB_OUTPUT_A     0x00       // commad bit for output A
#define CB_OUTPUT_B     0x80       // commad bit for output A
#define CB_GAIN_1X      0x20       // commad bit for gain 1x
#define CB_GAIN_2X      0x00       // commad bit for gain 2x
#define CB_SHDN         0x10       // commad bit for Output enabled
#define CB_NSHDN        0x00       // commad bit for Output buffer disabled, output is high-impedance

//=============================================================================
// Functions Declaration
//=============================================================================

/** Interface to the 12-Bit DAC with internal Vref and SPI interface
 *
  *  Using the driver:
 *   - remenber to setup SPI in main routine or use pins instance.
 *
 *  Defaults in this driver on start up:
 *   - Datasheet start up
 */
class MCP4822 {
public:
    /** Create an instance of the MCP4822 connected via specfied SPI instance.
     *
     * @param spi The mbed SPI instance (make in main routine)
     * @param nLDAC The Latch DAC Synchronization pin 
     * @param nCs The SPI chip select pin.
     */
    MCP4822(SPI& spi, PinName nLDAC, PinName nCs);
    
    /** Create an instance of the MCP4261 connected with SPI pins.
     *
     * @param nLDAC The Latch DAC Synchronization pin 
     * @param mosi The SPI Master Output, Slave Input pin.
     * @param miso The SPI Master Input, Slave Output pin. 
     * @param sck The SPI Serial Clock pin.
     * @param nCs The SPI chip select pin.
     */
    MCP4822(PinName nLDAC, PinName mosi, PinName miso,PinName sck, PinName nCs);
    

    /** Write to output A.
     *
     * @param gain the gain is 2x = '0' and 1x = '1'.
     * @param data The 12 bits value to write to the output.
     */
    void a(bool gain, int data);
    
    /** Write to output B.
     *
     * @param gain the gain is 2x = '0' and 1x = '1'.
     * @param data The 12 bits value to write to the output.
     */
    void b(bool gain, int data);
    
    /** Output Synchronization.
     *
     * @param act The LDAC is Active = true and Inactive = false.
     */
    void ldac(bool act);
    
    /** Output Shutdown.
     *
     * @param output A = '0' and B = '1'.
     * @param act The SHDN is Active = true and Inactive = false.
     */
    void shdn(bool output, bool act);
    
  

private:
    SPI& _spi;
    DigitalOut _nCs;
    DigitalOut _nLDAC;
    
    char _upper_half;
    char _lower_half;
    
    char _make_upper_half(bool output, bool gain, bool shdn, int data);
    char _make_lower_half(int data);
    void _write(bool output, bool gain, bool shdn, int data);             

};

#endif
#endif