/* SocketInterface Base Class
 * Copyright (c) 2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef SOCKET_INTERFACE_H
#define SOCKET_INTERFACE_H

#include "stdint.h"


/** Enum of socket protocols
 */
typedef enum {
    SOCK_TCP,    /*!< Socket connection over TCP */
    SOCK_UDP,    /*!< Socket connection over UDP */
} socket_protocol_t;


/** SocketInterface class
 *  Common interface for implementation specific sockets created through
 *  network interfaces. This class is used internally by the
 *  TCPSocket and UDPSocket classes
 */
class SocketInterface
{
public:
    /** Set the IP address of the socket
     *  @param ip IP address to connect to, copied internally
     */
    virtual void setIPAddress(const char *ip);

    /** Set the port of the socket
     *  @param port Port to connect to
     */
    virtual void setPort(uint16_t port);

    /** Open a connection to the underlying address
     *  @return 0 on success
     */
    virtual int32_t open() = 0;

    /** Close an open connection
     *  @return 0 on success
     */
    virtual int32_t close() = 0;

    /** Send data
     *  @param data Buffer of data to send
     *  @param len Size of data to send
     *  @param timeout_ms Maximum amount of time to wait
     *  @return 0 on success
     */
    virtual int32_t send(const void *data, uint32_t len, uint32_t timeout_ms) = 0;

    /** In client or server mode receive data
     *  @param data a buffer to store the data in
     *  @param amount The amount of data to receive
     *  @param timeout_ms The longest time to wait for the data
     *  @return Number of bytes sent or a negative value on failure
     */
    virtual int32_t recv(void *data, uint32_t len, uint32_t timeout_ms) = 0;
};

#endif
