/* Socket
 * Copyright (c) 2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef SOCKET_H
#define SOCKET_H

#include "NetworkInterface.h"

/** Abstract socket class
 *  API for handling general sockets. Supports IP address operations
 *  and sending/recieving data.
 */
class Socket
{
public:
    /** Set the URL of the socket
     *  Performs DNS lookup if necessary
     *  @param url URL to connect to
     *  @return 0 on success
     */
    int32_t setURL(const char *url);

    /** Set the IP address of the socket
     *  @param ip IP address to connect to, copied internally
     */
    void setIPAddress(const char *ip);

    /** Set the port of the socket
     *  @param port Port to connect to
     */
    void setPort(uint16_t port);

    /** Gets the IP address
     *  @return IP address to connect to
     */
    const char *getIPAddress() const;

    /** Gets the port
     *  @return Port to connect to
     */
    uint16_t getPort() const;


    /** Open a connection to the underlying address
     *  @param url Optional URL or IP address to connect to
     *  @param port Optional port to connect to
     *  @return 0 on success
     */
    int32_t open(const char *url = 0, uint16_t port = 0);

    /** Close an open connection
     *  @return 0 on success
     */
    int32_t close();

    /** Send data over the socket
     *  @param data Buffer of data to send
     *  @param len Size of data to send
     *  @param timeout_ms Maximum amount of time to wait
     *  @return 0 on success
     */
    int32_t send(const void *data, uint32_t len, uint32_t timeout_ms = 15000);

    /** Recieve data over the socket
     *  @param data Buffer to store recieved data
     *  @param len Size of provided buffer
     *  @param timeout_ms Maximum amount of time to wait
     *  @return Number of bytes sent or a negative value on failure
     */
    int32_t recv(void *data, uint32_t len, uint32_t timeout_ms = 15000);


protected:
    Socket(NetworkInterface *iface, socket_protocol_t proto);
    ~Socket();

private:
    NetworkInterface *_iface;
    socket_protocol_t _proto;
    SocketInterface *_socket;

    char _ip_address[SOCK_IP_SIZE];
    uint16_t _port;
};

#endif
