/* Socket
 * Copyright (c) 2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "Socket.h"
#include <string.h>

Socket::Socket(NetworkInterface *iface, socket_protocol_t proto)
    : _iface(iface)
    , _proto(proto)
    , _socket(0)
{
    memset(_ip_address, 0, SOCK_IP_SIZE);
    _port = 0;
}

Socket::~Socket()
{
    if (_socket) close();
}

int32_t Socket::setURL(const char *url)
{
    int32_t err = _iface->getHostByName(url, _ip_address);
    if (err) return err;

    if (_socket) {
        _socket->setIPAddress(_ip_address);
    }

    return 0;
}

void Socket::setIPAddress(const char *ip)
{
    strcpy(_ip_address, ip);

    if (_socket) {
        _socket->setIPAddress(_ip_address);
    }
}

void Socket::setPort(uint16_t port)
{
    _port = port;

    if (_socket) {
        _socket->setPort(_port);
    }
}

const char *Socket::getIPAddress() const
{
    return _ip_address;
}

uint16_t Socket::getPort() const
{
    return _port;
}

int32_t Socket::open(const char *url, uint16_t port)
{
    if (_socket) {
        int32_t err = close();
        if (err) return err;
    }

    if (url) {
        int32_t err = setURL(url);
        if (err) return err;
    }

    if (port) {
        setPort(port);
    }


    _socket = _iface->createSocket(_proto);
    if (!_socket) return -2;

    if (_ip_address[0]) {
        _socket->setIPAddress(_ip_address);
    }

    if (_port) {
        _socket->setPort(_port);
    }


    int32_t err = _socket->open();
    if (err) {
        _iface->destroySocket(_socket);
    }

    return err;
}

int32_t Socket::close()
{
    if (!_socket) return 0;

    int32_t err = _socket->close();

    if (!err) {
        _iface->destroySocket(_socket);
    }

    return err;
}

int32_t Socket::send(const void *data, uint32_t len, uint32_t timeout_ms)
{
    if (!_socket) return -2;
    return _socket->send(data, len, timeout_ms);
}

int32_t Socket::recv(void *data, uint32_t len, uint32_t timeout_ms)
{
    if (!_socket) return -2;
    return _socket->recv(data, len, timeout_ms);
}

