/* NetworkInterface Base Class
 * Copyright (c) 2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef NETWORK_INTERFACE_H
#define NETWORK_INTERFACE_H

#include "stdint.h"
#include "SocketInterface.h"

#define SOCK_IP_SIZE 16


/** NetworkInterface class
 *  Common interface that is shared between all hardware that
 *  can connect to a network over IP.
 */
class NetworkInterface
{
public:
    /** Enables DHCP and clears any static address
     *  DHCP is enabled by default
     *  @return 0 on success
     */
    virtual int32_t useDHCP() = 0;

    /** Set the static IP address of the network interface
     *  @param ip Static IP address, copied internally
     */
    virtual void setIPAddress(const char *ip);

    /** Set the static network mask of the network interface
     *  @param mask Static network mask, copied internally
     */
    virtual void setNetworkMask(const char *mask);

    /** Set the static gateway of the network interface
     *  @param gateway Gateway address, copied internally
     */
    virtual void setGateway(const char *gateway);

    /** Get the IP address
     *  @return IP address of the interface
     */
    virtual const char *getIPAddress();

    /** Get the network mask
     *  @return Network mask of the interface
     */
    virtual const char *getNetworkMask();

    /** Get the gateway
     *  @return Gateway address of the interface
     */
    virtual const char *getGateway();

    /** Get the current MAC address
     *  @return String MAC address of the interface
     */
    virtual const char *getMACAddress() = 0;

    /** Get the current status of the interface
     *  @return true if connected
     */
    virtual bool isConnected(void);
    
    /** Looks up the specified host's IP address
     *  @param name URL of host
     *  @param ip Buffer to hold IP address, must be at least SOCK_IP_SIZE
     *  @return 0 on success
     */
    int32_t getHostByName(const char *name, char *ip);

protected:
    NetworkInterface();

    friend class Socket;

    /** Internally create a socket
     *  @param proto The type of socket to open, SOCK_TCP or SOCK_UDP
     *  @return The allocated socket
     */
    virtual SocketInterface *createSocket(socket_protocol_t proto) = 0;

    /** Internally destroy a socket
     *  @param socket An allocated SocketInterface
     *  @returns 0 on success
     */
    virtual void destroySocket(SocketInterface *socket) = 0;

private:
    char _ip_address[SOCK_IP_SIZE];
    char _network_mask[SOCK_IP_SIZE];
    char _gateway[SOCK_IP_SIZE];
};

#endif
