/* Copyright (C) 2012 mbed.org, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef UDPSOCKET_H
#define UDPSOCKET_H

#include "Socket.h"

/**
UDP Socket
*/
class UDPSocket : public Socket {
public:
    /** UDPSocket lifetime
    */
    UDPSocket();
    ~UDPSocket();
    
    /** Bind a UDP Server Socket to a specific port
    \param port The port to listen for incoming connections on
    \return 0 on success, negative on failure.
    */
    int bind(int port);
    
    /** Send a packet to a remote endpoint
    \param addr     The remote IP address
    \param remote   The remote port
    \param data     The packet to be sent
    \param size     The length of the packet to be sent
    \return the number of written bytes on success (>=0) or -1 on failure
    */
    int sendto(const char *addr, uint16_t port, const void *data, unsigned size);
    
    /** Receive a packet from a remote endpoint
    \param addr     The remote IP address
    \param buffer   The buffer for storing the incoming packet data
                    If a packet is too long to fit in the supplied buffer,
                    excess bytes are discarded
    \param size     The length of the buffer
    \return the number of received bytes on success, negative on failure
    */
    int recvfrom(const char *addr, uint16_t port, void *buffer, unsigned size);

    /** Register a callback on when send is ready
    \param callback Function to call when send will succeed, may be called in
                    interrupt context.
    */
    void attach_send(FuncPtr<void()> callback);

    /** Register a callback on when recv is ready
    \param callback Function to call when recv will succeed, may be called in
                    interrupt context.
    */
    void attach_recv(FuncPtr<void()> callback);
};

#endif
