/* mbed Microcontroller Library
 * Copyright (c) 2018 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef MBED_NFC_REMOTE_INITIATOR_H
#define MBED_NFC_REMOTE_INITIATOR_H

#include <stdint.h>

#include "NFCDefinitions.h"
#include "NFCRemoteEndpoint.h"
#include "NFCNDEFCapable.h"
#include "stack/tech/iso7816/iso7816_app.h"

#include "platform/Span.h"

namespace mbed {
namespace nfc {

/**
 * @addtogroup nfc
 * @{
 */

class NFCController;

/**
 * This class represents a remote NFC initiator (the local controller being in target mode).
 *
 * An initiator can be a NFC reader, a NFC-enabled phone or other NFC device capable of generating a RF field.
 */
class NFCRemoteInitiator : public NFCRemoteEndpoint, public NFCNDEFCapable {
public:
    /**
     * Create a NFCRemoteInitiator.
     * @param[in] controller the NFCController instance that detected this endpoint
     * @param[in] buffer a bytes array used to store NDEF messages
     */
    NFCRemoteInitiator(NFCController *controller, const Span<uint8_t> &buffer);
    virtual ~NFCRemoteInitiator();

    /**
     * The NFCRemoteInitiator delegate. Users of the NFCRemoteInitiator class need to implement this delegate's methods to receive events.
     */
    struct Delegate : NFCRemoteEndpoint::Delegate, NFCNDEFCapable::Delegate {
    protected:
        ~Delegate() {}
    };

    /**
     * Set the delegate that will receive events generated by the initiator.
     *
     * @param[in] delegate the delegate instance to use
     */
    void set_delegate(Delegate *delegate);

    /**
     * Retrieve the NFC tag type exposed by the controller to communicate with the initiator.
     *
     * @return the relevant NFC tag type
     */
    virtual nfc_tag_type_t nfc_tag_type() const = 0;

    /**
     * Retrieve whether ISO7816 applications are supported by the underlying technology.
     *
     * @return whether ISO7816 applications are supported
     */
    virtual bool is_iso7816_supported() const = 0;

    /**
     * Register an ISO7816 application to be used by the initiator.
     *
     * @param[in] application a pointer to an nfc_tech_iso7816_app_t instance as defined by the MuNFC stack
     */
    virtual void add_iso7816_application(nfc_tech_iso7816_app_t *application) = 0;

protected:
    virtual void connected();
    virtual void disconnected();

private:
    // NFCNDEFCapable implementation
    virtual NFCNDEFCapable::Delegate *ndef_capable_delegate();

    Delegate *_delegate;
};

/**
 * @}
 */

} // namespace nfc
} // namespace mbed

#endif
