/**
 * @section LICENSE
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License 
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License along 
 * with this program; if not, write to the Free Software Foundation, Inc., 
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * @section DESCRIPTION
 * Library to interface with the SCP1000 pressure and temperature sensor.
 */

#ifndef _SCP1000_H
#define _SCP1000_H

#include "mbed.h"

/**
 * Class to interface with the SCP1000 pressure and temperature sensor.
 */
class SCP1000 {
    public:
        /**
         * Constructor.
         *
         * @param mosi SPI MOSI pin
         * @param miso SPI MISO pin
         * @param sclk SPI SCLK pin
         * @param cs Chip select pin
         */
        SCP1000(PinName mosi, PinName miso, PinName sclk, PinName cs);
        
        ~SCP1000() { /* empty */ };
        
        /**
         * Read the pressure.
         *
         * @returns The pressure in pascals.
         */
        unsigned long readPressure();
        
        /**
         * Read the temperature.
         *
         * @returns The temperature in Celsius.
         */
        float readTemperature();
        
        
    private:
        static const char PRESSURE = 0x1F;   //Pressure 3 MSB
        static const char PRESSURE_LSB = 0x20; //Pressure 16 LSB
        static const char TEMP = 0x21;       //16 bit temp
        SPI m_spi;
        DigitalOut m_cs;
        
        char read_register(char register_name);
        void write_register(char register_name, char register_value);
        float read_register16(char register_name);

};

#endif // _SCP1000_H