/* Copyright (c) 2010-2011 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#pragma once

#include "USBDevice.h"
#include "CircBuffer.h"

#if defined(TARGET_LPC1768)||defined(TARGET_LPC4088)
#define CDC_EPINT_IN   EP1IN
#define CDC_EPBULK_IN  EP5IN 
#define CDC_EPBULK_OUT EP5OUT
#elif defined(TARGET_LPC1347)||defined(TARGET_LPC11U24)||defined(TARGET_LPC11U35_401)||defined(TARGET_LPC11U35_501)||defined(TARGET_LPC1549)||defined(TARGET_LPC11U68)
#define CDC_EPINT_IN   EP1IN
#define CDC_EPBULK_IN  EP3IN 
#define CDC_EPBULK_OUT EP3OUT
#elif defined(TARGET_KL25Z)||defined(TARGET_KL46Z)
#define CDC_EPINT_IN   EP1IN
#define CDC_EPBULK_IN  EP5IN 
#define CDC_EPBULK_OUT EP5OUT
#else
#error "target type error"
#endif

class USB_CDC {
public:
    USB_CDC(USBDevice* device);

    /** target to virtual COM
     */
    void putc(int c);
    
    /** virtial COM to target
     */
    int getc();

    int readable();
    
    int writeable();

    /*
    * Send a buffer
    *
    * @param endpoint endpoint which will be sent the buffer
    * @param buffer buffer to be sent
    * @param size length of the buffer
    * @returns true if successful
    */
    bool send(uint8_t * buffer, uint32_t size);
    
    /*
    * Read a buffer from a certain endpoint. Warning: blocking
    *
    * @param endpoint endpoint to read
    * @param buffer buffer where will be stored bytes
    * @param size the number of bytes read will be stored in *size
    * @param maxSize the maximum length that can be read
    * @returns true if successful
    */
    bool readEP(uint8_t * buffer, uint32_t * size);
    
    /*
    * Read a buffer from a certain endpoint. Warning: non blocking
    *
    * @param endpoint endpoint to read
    * @param buffer buffer where will be stored bytes
    * @param size the number of bytes read will be stored in *size
    * @param maxSize the maximum length that can be read
    * @returns true if successful
    */
    bool readEP_NB(uint8_t * buffer, uint32_t * size);

    bool Request_callback(CONTROL_TRANSFER* transfer);
    bool RequestCompleted_callback(CONTROL_TRANSFER* transfer, uint8_t* buf, int length);
    bool EPBULK_OUT_callback();

    /**
     * Attach a callback to call when serial's settings are changed.
     *
     * @param fptr function pointer
     */
    void attach(void (*fptr)(int baud, int bits, int parity, int stop)) {
        settingsChangedCallback = fptr;
    }

    void attachControlLineStateChanged(void (*fptr)(int rts, int dtr)) {
    	controlLineStateChangedCallback = fptr;
    }

    void attachSendBreak(void (*fptr)(uint16_t duration)) {
    	sendBreakCallback = fptr;
    }
    
protected:
    void lineCodingChanged(int baud, int bits, int parity, int stop);
    void controlLineStateChanged(int rts, int dtr);
    void sendBreak(uint16_t duration);

private:
    USBDevice* _device;
    CircBuffer<uint8_t> _rx_buf;
    void (*settingsChangedCallback)(int baud, int bits, int parity, int stop);
    void (*controlLineStateChangedCallback)(int rts, int dtr);
    void (*sendBreakCallback)(uint16_t duration);
    volatile bool terminal_connected;
};

